"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.tlog = exports.TelemetryLoggerImpl = void 0;
var _helpers = require("./helpers");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * This custom logger extends the base kibana Logger with the following functionality:
 *  - Exposes a helper `TelemetryLogger::l` method that logs at the
 *    info or debug level depending on whether the instance is a cloud deployment or not.
 *  - For the above method as well as the regular debug, info, warn, error, etc.
 *    it includes the cluster uuid and name as part of the metadata structured fields.
 *
 * Please try to use a meaningful logger name, e.g.:
 *
 * ```js
 * const log = new TelemetryLoggerImpl(logger.get('tasks.endpoint'), ...);
 * ````
 * instead of
 *
 * ```js
 * const log = new TelemetryLoggerImpl(logger, ...);
 * ````
 *
 * It makes easier to browse the logs by filtering by the structured argument `logger`.
 */
class TelemetryLoggerImpl {
  constructor(delegate, mdc) {
    this.delegate = delegate;
    this.mdc = mdc;
  }
  l(message, meta) {
    if (_helpers.isElasticCloudDeployment) {
      this.info(message, meta);
    } else {
      this.debug(message, meta);
    }
  }
  trace(message, meta) {
    this.delegate.trace(message, logMeta(meta, this.mdc));
  }
  debug(message, meta) {
    this.delegate.debug(message, logMeta(meta, this.mdc));
  }
  info(message, meta) {
    this.delegate.info(message, logMeta(meta, this.mdc));
  }
  warn(errorOrMessage, meta) {
    this.delegate.warn(errorOrMessage, logMeta(meta, this.mdc));
  }
  error(errorOrMessage, meta) {
    this.delegate.error(errorOrMessage, logMeta(meta, this.mdc));
  }
  fatal(errorOrMessage, meta) {
    this.delegate.fatal(errorOrMessage, logMeta(meta, this.mdc));
  }
  log(record) {
    this.delegate.log(record);
  }
  isLevelEnabled(level) {
    return this.delegate.isLevelEnabled(level);
  }
  get(...childContextPaths) {
    const logger = this.delegate.get(...childContextPaths);
    return new TelemetryLoggerImpl(logger, this.mdc);
  }
}
exports.TelemetryLoggerImpl = TelemetryLoggerImpl;
const tlog = (logger, message, meta) => {
  if (_helpers.isElasticCloudDeployment) {
    logger.info(message, logMeta(meta));
  } else {
    logger.debug(message, logMeta(meta));
  }
};

// helper method to merge a given LogMeta with the cluster info (if exists)
exports.tlog = tlog;
function logMeta(meta, mdc) {
  const clusterInfoMeta = _helpers.clusterInfo ? {
    cluster_uuid: _helpers.clusterInfo === null || _helpers.clusterInfo === void 0 ? void 0 : _helpers.clusterInfo.cluster_uuid,
    cluster_name: _helpers.clusterInfo === null || _helpers.clusterInfo === void 0 ? void 0 : _helpers.clusterInfo.cluster_name
  } : {};
  return {
    ...clusterInfoMeta,
    ...(meta !== null && meta !== void 0 ? meta : {}),
    ...(mdc !== null && mdc !== void 0 ? mdc : {})
  };
}