"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getUniquePrebuiltRuleIds = exports.getPrebuiltRulesForMigration = exports.getPrebuiltRules = void 0;
var _prebuilt_rule_objects_client = require("../../../../detection_engine/prebuilt_rules/logic/rule_objects/prebuilt_rule_objects_client");
var _fetch_rule_versions_triad = require("../../../../detection_engine/prebuilt_rules/logic/rule_versions/fetch_rule_versions_triad");
var _prebuilt_rule_assets_client = require("../../../../detection_engine/prebuilt_rules/logic/rule_assets/prebuilt_rule_assets_client");
var _convert_prebuilt_rule_asset_to_rule_response = require("../../../../detection_engine/rule_management/logic/detection_rules_client/converters/convert_prebuilt_rule_asset_to_rule_response");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getUniquePrebuiltRuleIds = migrationRules => {
  const rulesIds = new Set();
  migrationRules.forEach(rule => {
    var _rule$elastic_rule;
    if ((_rule$elastic_rule = rule.elastic_rule) !== null && _rule$elastic_rule !== void 0 && _rule$elastic_rule.prebuilt_rule_id) {
      rulesIds.add(rule.elastic_rule.prebuilt_rule_id);
    }
  });
  return Array.from(rulesIds);
};
exports.getUniquePrebuiltRuleIds = getUniquePrebuiltRuleIds;
/**
 * Gets Elastic prebuilt rules
 * @param rulesClient The rules client to fetch prebuilt rules
 * @param savedObjectsClient The saved objects client
 * @param rulesIds The list of IDs to filter requested prebuilt rules. If not specified, all available prebuilt rules will be returned.
 * @returns
 */
const getPrebuiltRules = async (rulesClient, savedObjectsClient, rulesIds) => {
  const ruleAssetsClient = (0, _prebuilt_rule_assets_client.createPrebuiltRuleAssetsClient)(savedObjectsClient);
  const ruleObjectsClient = (0, _prebuilt_rule_objects_client.createPrebuiltRuleObjectsClient)(rulesClient);
  const prebuiltRulesMap = await (0, _fetch_rule_versions_triad.fetchRuleVersionsTriad)({
    ruleAssetsClient,
    ruleObjectsClient
  });

  // Filter out prebuilt rules by `rule_id`
  let filteredPrebuiltRulesMap;
  if (rulesIds) {
    filteredPrebuiltRulesMap = new Map();
    for (const ruleId of rulesIds) {
      const prebuiltRule = prebuiltRulesMap.get(ruleId);
      if (prebuiltRule) {
        filteredPrebuiltRulesMap.set(ruleId, prebuiltRule);
      }
    }
  } else {
    filteredPrebuiltRulesMap = prebuiltRulesMap;
  }
  const prebuiltRules = {};
  filteredPrebuiltRulesMap.forEach((ruleVersions, ruleId) => {
    if (ruleVersions.target) {
      prebuiltRules[ruleId] = {
        target: (0, _convert_prebuilt_rule_asset_to_rule_response.convertPrebuiltRuleAssetToRuleResponse)(ruleVersions.target),
        current: ruleVersions.current
      };
    }
  });
  return prebuiltRules;
};

/**
 * Gets Elastic prebuilt rules
 * @param migrationId The `id` of the migration to get related prebuilt rules for
 * @param ruleMigrationsClient The rules migration client to migration rules data
 * @param rulesClient The rules client to fetch prebuilt rules
 * @param savedObjectsClient The saved objects client
 * @returns
 */
exports.getPrebuiltRules = getPrebuiltRules;
const getPrebuiltRulesForMigration = async (migrationId, ruleMigrationsClient, rulesClient, savedObjectsClient) => {
  const options = {
    filters: {
      prebuilt: true
    }
  };
  const batches = ruleMigrationsClient.data.items.searchBatches(migrationId, options);
  const rulesIds = new Set();
  let results = await batches.next();
  while (results.length) {
    results.forEach(rule => {
      var _rule$elastic_rule2;
      if ((_rule$elastic_rule2 = rule.elastic_rule) !== null && _rule$elastic_rule2 !== void 0 && _rule$elastic_rule2.prebuilt_rule_id) {
        rulesIds.add(rule.elastic_rule.prebuilt_rule_id);
      }
    });
    results = await batches.next();
  }
  const prebuiltRulesIds = Array.from(rulesIds);
  const prebuiltRules = await getPrebuiltRules(rulesClient, savedObjectsClient, prebuiltRulesIds);
  return prebuiltRules;
};
exports.getPrebuiltRulesForMigration = getPrebuiltRulesForMigration;