"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createPrivilegedUsersCrudService = void 0;
var _lodash = require("lodash");
var _esQuery = require("@kbn/es-query");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const createPrivilegedUsersCrudService = ({
  deps,
  index
}) => {
  const esClient = deps.clusterClient.asCurrentUser;
  const create = async (user, source, maxUsersAllowed) => {
    var _user$user;
    deps.logger.info(`Maximum supported number of privileged users allowed: ${maxUsersAllowed}`);
    // Check if user already exists by username
    const username = (_user$user = user.user) === null || _user$user === void 0 ? void 0 : _user$user.name;
    if (username) {
      const existingUserResponse = await esClient.search({
        index,
        query: {
          term: {
            'user.name': username
          }
        },
        size: 1
      });
      if (existingUserResponse.hits.hits.length > 0) {
        const existingUser = existingUserResponse.hits.hits[0];
        const existingUserId = existingUser._id;
        if (existingUserId) {
          var _existingUserDoc$labe;
          const existingUserDoc = existingUser._source;
          const existingSources = (existingUserDoc === null || existingUserDoc === void 0 ? void 0 : (_existingUserDoc$labe = existingUserDoc.labels) === null || _existingUserDoc$labe === void 0 ? void 0 : _existingUserDoc$labe.sources) || [];
          const updatedSources = existingSources.includes(source) ? existingSources : [...existingSources, source];
          await esClient.update({
            index,
            id: existingUserId,
            refresh: 'wait_for',
            doc: {
              ...user,
              user: {
                ...user.user,
                is_privileged: true,
                entity: {
                  attributes: {
                    Privileged: true
                  }
                }
              },
              labels: {
                sources: updatedSources
              }
            }
          });
          const updatedUser = await get(existingUserId);
          if (!updatedUser) {
            throw new Error(`Failed to retrieve updated user: ${existingUserId}`);
          }
          return updatedUser;
        }
      }
    }

    // Check user count limit before creating new user
    const currentUserCount = await esClient.count({
      index,
      query: {
        term: {
          'user.is_privileged': true
        }
      }
    });
    if (currentUserCount.count >= maxUsersAllowed) {
      throw new Error(`Cannot create user: Maximum user limit of ${maxUsersAllowed} reached`);
    }

    // Create new user
    const doc = (0, _lodash.merge)(user, {
      '@timestamp': new Date().toISOString(),
      user: {
        is_privileged: true,
        entity: {
          attributes: {
            Privileged: true
          }
        }
      },
      labels: {
        sources: [source]
      }
    });
    const res = await esClient.index({
      index,
      refresh: 'wait_for',
      document: doc
    });
    const newUser = await get(res._id);
    if (!newUser) {
      throw new Error(`Failed to create user: ${res._id}`);
    }
    return newUser;
  };
  const get = async id => {
    const response = await esClient.get({
      index,
      id
    });
    return response.found ? {
      ...response._source,
      id: response._id
    } : undefined;
  };
  const update = async (id, user) => {
    await esClient.update({
      index,
      refresh: 'wait_for',
      id,
      doc: user
    });
    return get(id);
  };
  const list = async kuery => {
    const query = kuery ? (0, _esQuery.toElasticsearchQuery)((0, _esQuery.fromKueryExpression)(kuery)) : {
      match_all: {}
    };
    const response = await esClient.search({
      size: 10000,
      index,
      query
    });
    return response.hits.hits.map(hit => ({
      id: hit._id,
      ...hit._source
    }));
  };
  const _delete = async id => {
    await esClient.delete({
      index,
      id
    });
  };
  return {
    create,
    get,
    update,
    list,
    delete: _delete
  };
};
exports.createPrivilegedUsersCrudService = createPrivilegedUsersCrudService;