"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createPrivmonIndexService = void 0;
var _create_or_update_index = require("../../../utils/create_or_update_index");
var _mappings = require("./mappings");
var _pipeline = require("./pipeline");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const createPrivmonIndexService = dataClient => {
  const {
    deps,
    index
  } = dataClient;
  const internalUserClient = deps.clusterClient.asInternalUser;
  const _upsertIndex = async () => {
    dataClient.log('debug', `Creating or updating index: ${index}`);
    await (0, _create_or_update_index.createOrUpdateIndex)({
      esClient: internalUserClient,
      logger: deps.logger,
      options: {
        index,
        mappings: (0, _mappings.generateUserIndexMappings)(),
        settings: {
          hidden: true,
          mode: 'lookup',
          default_pipeline: _pipeline.PRIVMON_EVENT_INGEST_PIPELINE_ID
        }
      }
    }).catch(e => {
      if (e.meta.body.error.type === 'resource_already_exists_exception') {
        dataClient.log('debug', 'Caught error: Privilege monitoring index already exists. Continuing workflow');
        return;
      }
      throw e;
    });
  };
  const doesIndexExist = async () => {
    return internalUserClient.indices.exists({
      index
    }).catch(e => {
      dataClient.log('debug', `Error checking if index exists: ${e.message}`);
      return false;
    });
  };
  const _createIngestPipelineIfDoesNotExist = async () => {
    const pipelinesResponse = await internalUserClient.ingest.getPipeline({
      id: _pipeline.PRIVMON_EVENT_INGEST_PIPELINE_ID
    }, {
      ignore: [404]
    });
    if (pipelinesResponse[_pipeline.PRIVMON_EVENT_INGEST_PIPELINE_ID]) {
      dataClient.log('info', 'Privileged user monitoring ingest pipeline already exists.');
      return;
    }
    dataClient.log('info', 'Privileged user monitoring ingest pipeline does not exist, creating.');
    await internalUserClient.ingest.putPipeline(_pipeline.eventIngestPipeline);
  };
  const initialisePrivmonIndex = async () => {
    await _upsertIndex();
    await _createIngestPipelineIfDoesNotExist();
  };
  return {
    _upsertIndex,
    _createIngestPipelineIfDoesNotExist,
    initialisePrivmonIndex,
    doesIndexExist
  };
};
exports.createPrivmonIndexService = createPrivmonIndexService;