"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.EntityStoreCrudClient = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _uuid = require("uuid");
var _std = require("@kbn/std");
var _entitiesSchema = require("@kbn/entities-schema");
var _types = require("../../../../common/entity_analytics/types");
var _errors = require("./errors");
var _utils = require("./utils");
var _build_update_script = require("./painless/build_update_script");
var _updates_entity_data_stream = require("./elasticsearch_assets/updates_entity_data_stream");
var _engine_description = require("./installation/engine_description");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const ENTITY_ID_FIELD = 'entity.id';
class EntityStoreCrudClient {
  constructor({
    clusterClient,
    namespace,
    logger,
    dataClient
  }) {
    (0, _defineProperty2.default)(this, "esClient", void 0);
    (0, _defineProperty2.default)(this, "namespace", void 0);
    (0, _defineProperty2.default)(this, "logger", void 0);
    (0, _defineProperty2.default)(this, "dataClient", void 0);
    this.esClient = clusterClient.asCurrentUser;
    this.namespace = namespace;
    this.logger = logger;
    this.dataClient = dataClient;
  }
  async upsertEntity(type, doc, force = false) {
    await this.assertEngineIsRunning(type);
    await this.assertCRUDApiIsEnabled(type);
    const normalizedDocToECS = normalizeToECS(doc);
    const flatProps = (0, _std.getFlattenedObject)(normalizedDocToECS);
    const entityTypeDescription = _engine_description.engineDescriptionRegistry[type];
    const fieldDescriptions = getFieldDescriptions(flatProps, entityTypeDescription);
    if (!force) {
      assertOnlyNonForcedAttributesInReq(fieldDescriptions);
    }
    this.logger.info(`Updating entity '${doc.entity.id}' (type ${type})`);
    const painlessUpdate = (0, _build_update_script.buildUpdateEntityPainlessScript)(fieldDescriptions);
    if (!painlessUpdate) {
      throw new _errors.BadCRUDRequestError(`The request doesn't contain any update`);
    }
    const updateByQueryResp = await this.esClient.updateByQuery({
      index: (0, _utils.getEntitiesIndexName)(type, this.namespace),
      query: {
        term: {
          'entity.id': doc.entity.id
        }
      },
      script: {
        source: painlessUpdate,
        lang: 'painless'
      },
      conflicts: 'proceed'
    });
    if (updateByQueryResp.version_conflicts) {
      throw new _errors.DocumentVersionConflictError();
    }
    await this.esClient.create({
      id: (0, _uuid.v4)(),
      index: (0, _updates_entity_data_stream.getEntityUpdatesDataStreamName)(type, this.namespace),
      document: buildDocumentToUpdate(type, normalizedDocToECS)
    });
  }
  async assertEngineIsRunning(type) {
    const engineRunning = await this.dataClient.isEngineRunning(_types.EntityType[type]);
    if (!engineRunning) {
      throw new _errors.EngineNotRunningError(type);
    }
  }
  async assertCRUDApiIsEnabled(type) {
    const enabled = await this.dataClient.isCapabilityEnabled(_types.EntityType[type], _entitiesSchema.EntityStoreCapability.CRUD_API);
    if (!enabled) {
      throw new _errors.CapabilityNotEnabledError(_entitiesSchema.EntityStoreCapability.CRUD_API);
    }
  }
}
exports.EntityStoreCrudClient = EntityStoreCrudClient;
function assertOnlyNonForcedAttributesInReq(fields) {
  const notAllowedProps = [];
  for (const [name, description] of Object.entries(fields)) {
    if (!description.allowAPIUpdate && name !== ENTITY_ID_FIELD) {
      notAllowedProps.push(name);
    }
  }
  if (notAllowedProps.length > 0) {
    const notAllowedPropsString = notAllowedProps.join(', ');
    throw new _errors.BadCRUDRequestError(`The following attributes are not allowed to be ` + `updated without forcing it (?force=true): ${notAllowedPropsString}`);
  }
}
function normalizeToECS(doc) {
  const normalizedDoc = {
    ...doc
  }; // create copy

  const {
    attributes,
    lifecycle,
    behaviors,
    relationships
  } = doc.entity;
  const objsToCheck = {
    attributes,
    lifecycle,
    behaviors,
    relationships
  };
  for (const key of Object.keys(objsToCheck)) {
    const typedKey = key;
    const value = objsToCheck[typedKey];
    if (value) {
      normalizedDoc.entity[typedKey] = convertFieldsToCustomECS(value);
    }
  }
  return normalizedDoc;
}
function convertFieldsToCustomECS(obj) {
  if (!obj) {
    return undefined;
  }
  const customECSFormatObj = {};
  const keys = Object.keys(obj);
  for (const key of keys) {
    customECSFormatObj[convertToECSCustomFormat(key)] = obj[key];
  }
  return customECSFormatObj;
}
function convertToECSCustomFormat(key) {
  // assuming we are getting a key in snake case;
  return key.charAt(0).toUpperCase() + key.slice(1);
}
function buildDocumentToUpdate(type, data) {
  var _data$entity;
  const now = new Date().toISOString();
  if (type === 'generic') {
    return {
      '@timestamp': now,
      ...data
    };
  }

  // Get host, user, service field
  const typeData = data[type] || {};

  // Force name to be picked by the store
  typeData.name = (_data$entity = data.entity) === null || _data$entity === void 0 ? void 0 : _data$entity.id;
  // Nest entity under type data
  typeData.entity = data.entity;
  const doc = {
    '@timestamp': now,
    ...data
  };

  // Remove entity from root
  delete doc.entity;

  // override the host, user service
  // field with the built value
  doc[type] = typeData;
  return doc;
}
function getFieldDescriptions(flatProps, description) {
  const allFieldDescriptions = description.fields.reduce((obj, field) => {
    obj[field.destination || field.source] = field;
    return obj;
  }, {});
  const invalid = [];
  const descriptions = {};
  for (const [key, value] of Object.entries(flatProps)) {
    if (key === ENTITY_ID_FIELD || key === description.identityField) {
      // eslint-disable-next-line no-continue
      continue;
    }
    if (!allFieldDescriptions[key]) {
      invalid.push(key);
    } else {
      descriptions[key] = {
        ...allFieldDescriptions[key],
        value
      };
    }
  }

  // This will catch differences between
  // API and entity store definition
  if (invalid.length > 0) {
    const invalidString = invalid.join(', ');
    throw new _errors.BadCRUDRequestError(`The following attributes are not allowed to be updated: ${invalidString}`);
  }
  return descriptions;
}