"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getSignalIdToMatchedQueriesMap = getSignalIdToMatchedQueriesMap;
var _lodash = require("lodash");
var _types = require("./types");
var _get_threat_list = require("./get_threat_list");
var _utils = require("./utils");
var _enrich_signal_threat_matches = require("./enrich_signal_threat_matches");
var _build_threat_mapping_filter = require("./build_threat_mapping_filter");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const addMatchedQueryToMaps = ({
  signalId,
  threatHit,
  decodedQuery,
  signalIdToMatchedQueriesMap,
  maxThreatsReachedMap
}) => {
  const signalMatch = signalIdToMatchedQueriesMap.get(signalId);
  const threatQuery = {
    threatMappingIndex: decodedQuery.threatMappingIndex,
    queryType: decodedQuery.queryType
  };
  if (!signalMatch) {
    signalIdToMatchedQueriesMap.set(signalId, [{
      threatHit,
      query: threatQuery
    }]);
    return;
  }
  if (signalMatch.length === _enrich_signal_threat_matches.MAX_NUMBER_OF_SIGNAL_MATCHES) {
    maxThreatsReachedMap.set(signalId, true);
  } else if (signalMatch.length < _enrich_signal_threat_matches.MAX_NUMBER_OF_SIGNAL_MATCHES) {
    signalMatch.push({
      threatHit,
      query: threatQuery
    });
  }
};

/**
 * fetches threats and creates a map that matches each signal with a list of threat queries
 * @param options.fieldAndValueToDocIdsMap - map of signal values from terms query results
 */
async function getSignalIdToMatchedQueriesMap({
  allowedFieldsForTermsQuery,
  pitId,
  reassignThreatPitId,
  services,
  sharedParams,
  signals,
  threatFilters,
  threatIndexFields,
  threatIndicatorPath
}) {
  var _threatFiltersFromEve;
  let threatList;
  const signalIdToMatchedQueriesMap = new Map();
  // number of threat matches per signal is limited by MAX_NUMBER_OF_SIGNAL_MATCHES. Once it hits this number, threats stop to be processed for a signal
  const maxThreatsReachedMap = new Map();
  const threatMappings = sharedParams.completeRule.ruleParams.threatMapping;
  const threatFiltersFromEvents = (0, _build_threat_mapping_filter.buildThreatMappingFilter)({
    threatMappings,
    threatList: signals,
    entryKey: 'field',
    allowedFieldsForTermsQuery
  });
  if (!threatFiltersFromEvents.query || ((_threatFiltersFromEve = threatFiltersFromEvents.query) === null || _threatFiltersFromEve === void 0 ? void 0 : _threatFiltersFromEve.bool.should.length) === 0) {
    // empty event list and we do not want to return everything as being
    // a hit so opt to return the existing result.
    return signalIdToMatchedQueriesMap;
  }
  const threatMatchedFields = (0, _utils.getMatchedFields)(sharedParams.completeRule.ruleParams.threatMapping);
  const threatSearchParams = {
    sharedParams,
    esClient: services.scopedClusterClient.asCurrentUser,
    threatFilters: [...threatFilters, threatFiltersFromEvents],
    threatListConfig: {
      _source: [...threatMatchedFields.threat, `${threatIndicatorPath}.*`, 'threat.feed.*'],
      fields: undefined
    },
    pitId,
    reassignPitId: reassignThreatPitId,
    indexFields: threatIndexFields
  };
  const fieldAndValueToDocIdsMap = (0, _utils.getFieldAndValueToDocIdsMap)({
    eventList: signals,
    threatMatchedFields
  });
  threatList = await (0, _get_threat_list.getThreatList)({
    ...threatSearchParams,
    searchAfter: undefined
  });
  while (maxThreatsReachedMap.size < signals.length && ((_threatList = threatList) === null || _threatList === void 0 ? void 0 : _threatList.hits.hits.length) > 0) {
    var _threatList;
    threatList.hits.hits.forEach(threatHit => {
      const matchedQueries = Array.isArray(threatHit === null || threatHit === void 0 ? void 0 : threatHit.matched_queries) ? threatHit.matched_queries : [];
      matchedQueries.forEach(matchedQuery => {
        const decodedQuery = (0, _utils.decodeThreatMatchNamedQuery)(matchedQuery);
        const threatMappingIndex = decodedQuery.threatMappingIndex;
        const threatMapping = threatMappings[threatMappingIndex];
        if (decodedQuery.queryType === _types.ThreatMatchQueryType.term) {
          const threatValue = (0, _lodash.get)(threatHit === null || threatHit === void 0 ? void 0 : threatHit._source, threatMapping.entries[0].value);
          const values = Array.isArray(threatValue) ? threatValue : [threatValue];
          values.forEach(value => {
            if (value && fieldAndValueToDocIdsMap) {
              const ids = fieldAndValueToDocIdsMap[threatMapping.entries[0].field][value === null || value === void 0 ? void 0 : value.toString()];
              ids === null || ids === void 0 ? void 0 : ids.forEach(id => {
                addMatchedQueryToMaps({
                  signalId: id,
                  threatHit,
                  decodedQuery,
                  signalIdToMatchedQueriesMap,
                  maxThreatsReachedMap
                });
              });
            }
          });
        } else {
          const signalId = decodedQuery.id;
          if (!signalId) {
            return;
          }
          addMatchedQueryToMaps({
            signalId,
            threatHit,
            decodedQuery,
            signalIdToMatchedQueriesMap,
            maxThreatsReachedMap
          });
        }
      });
    });
    threatList = await (0, _get_threat_list.getThreatList)({
      ...threatSearchParams,
      searchAfter: threatList.hits.hits[threatList.hits.hits.length - 1].sort
    });
  }
  return signalIdToMatchedQueriesMap;
}