"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.handleCoverageOverviewRequest = handleCoverageOverviewRequest;
var _rule_filtering = require("../../../../../../../common/detection_engine/rule_management/rule_filtering");
var _detection_engine = require("../../../../../../../common/api/detection_engine");
var _find_rules = require("../../../logic/search/find_rules");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function handleCoverageOverviewRequest({
  params: {
    filter
  },
  deps: {
    rulesClient
  }
}) {
  var _filter$source$includ, _filter$source, _filter$source$includ2, _filter$source2;
  const activitySet = new Set(filter === null || filter === void 0 ? void 0 : filter.activity);
  const kqlFilter = (0, _rule_filtering.convertRulesFilterToKQL)({
    filter: filter === null || filter === void 0 ? void 0 : filter.search_term,
    showCustomRules: (_filter$source$includ = filter === null || filter === void 0 ? void 0 : (_filter$source = filter.source) === null || _filter$source === void 0 ? void 0 : _filter$source.includes(_detection_engine.CoverageOverviewRuleSource.Custom)) !== null && _filter$source$includ !== void 0 ? _filter$source$includ : false,
    showElasticRules: (_filter$source$includ2 = filter === null || filter === void 0 ? void 0 : (_filter$source2 = filter.source) === null || _filter$source2 === void 0 ? void 0 : _filter$source2.includes(_detection_engine.CoverageOverviewRuleSource.Prebuilt)) !== null && _filter$source$includ2 !== void 0 ? _filter$source$includ2 : false,
    enabled: getIsEnabledFilter(activitySet)
  });

  // rulesClient.find uses ES Search API to fetch the rules. It has some limitations when the number of rules exceeds
  // index.max_result_window (set to 10K by default) Kibana fails. A proper way to handle it is via ES PIT API.
  // This way the endpoint handles max 10K rules for now while support for the higher number of rules will be addressed
  // in https://github.com/elastic/kibana/issues/160698
  const rules = await (0, _find_rules.findRules)({
    rulesClient,
    filter: kqlFilter,
    fields: ['name', 'enabled', 'params.threat'],
    page: 1,
    perPage: 10000,
    sortField: undefined,
    sortOrder: undefined
  });
  return rules.data.reduce(appendRuleToResponse, {
    coverage: {},
    unmapped_rule_ids: [],
    rules_data: {}
  });
}
function getIsEnabledFilter(activitySet) {
  const bothSpecified = activitySet.has(_detection_engine.CoverageOverviewRuleActivity.Enabled) && activitySet.has(_detection_engine.CoverageOverviewRuleActivity.Disabled);
  const noneSpecified = !activitySet.has(_detection_engine.CoverageOverviewRuleActivity.Enabled) && !activitySet.has(_detection_engine.CoverageOverviewRuleActivity.Disabled);
  return bothSpecified || noneSpecified ? undefined : activitySet.has(_detection_engine.CoverageOverviewRuleActivity.Enabled);
}
function appendRuleToResponse(response, rule) {
  const categories = extractRuleMitreCategories(rule);
  for (const category of categories) {
    if (!response.coverage[category]) {
      response.coverage[category] = [rule.id];
    } else {
      response.coverage[category].push(rule.id);
    }
  }
  if (categories.length === 0) {
    response.unmapped_rule_ids.push(rule.id);
  }
  response.rules_data[rule.id] = {
    name: rule.name,
    activity: rule.enabled ? _detection_engine.CoverageOverviewRuleActivity.Enabled : _detection_engine.CoverageOverviewRuleActivity.Disabled
  };
  return response;
}

/**
 * Extracts rule's MITRE ATT&CK™ tactics, techniques and subtechniques
 *
 * @returns an array of MITRE ATT&CK™ tactics, techniques and subtechniques
 */
function extractRuleMitreCategories(rule) {
  if (!rule.params.threat) {
    return [];
  }

  // avoid duplications just in case data isn't valid in ES
  const categories = new Set();
  for (const threatItem of rule.params.threat) {
    if (threatItem.framework !== 'MITRE ATT&CK') {
      // eslint-disable-next-line no-continue
      continue;
    }
    categories.add(threatItem.tactic.id);
    for (const technique of (_threatItem$technique = threatItem.technique) !== null && _threatItem$technique !== void 0 ? _threatItem$technique : []) {
      var _threatItem$technique;
      categories.add(technique.id);
      for (const subtechnique of (_technique$subtechniq = technique.subtechnique) !== null && _technique$subtechniq !== void 0 ? _technique$subtechniq : []) {
        var _technique$subtechniq;
        categories.add(subtechnique.id);
      }
    }
  }
  return Array.from(categories);
}