"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.updateDeletedPolicyResponseActions = void 0;
var _services = require("../../endpoint/services");
var _utils = require("../../endpoint/utils");
var _constants = require("../../../common/endpoint/constants");
var _constants2 = require("../../../common/endpoint/service/response_actions/constants");
var _stringify = require("../../endpoint/utils/stringify");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const updateDeletedPolicyResponseActions = async (endpointService, deletedIntegrationPolicies) => {
  const logger = endpointService.createLogger('updateDeletedPolicyResponseActions');
  if (!endpointService.experimentalFeatures.endpointManagementSpaceAwarenessEnabled) {
    logger.debug(`Space awareness feature flag is disabled. Nothing to do`);
    return;
  }
  if (deletedIntegrationPolicies.length === 0) {
    logger.debug(`No deleted integration policies provided on input. Nothing to do`);
    return;
  }
  if (!(await endpointService.getInternalFleetServices(undefined, true).isEndpointPackageInstalled())) {
    logger.debug(`Endpoint package not installed. Nothing to do`);
    return;
  }
  logger.debug(`Checking if response action requests need to be updated for deleted integration policies`);
  const packageNames = Object.values(_constants2.RESPONSE_ACTIONS_SUPPORTED_INTEGRATION_TYPES).flat();
  const policyIds = [];
  for (const deletedPolicy of deletedIntegrationPolicies) {
    var _deletedPolicy$packag, _deletedPolicy$packag2;
    if (deletedPolicy.success && packageNames.includes((_deletedPolicy$packag = deletedPolicy === null || deletedPolicy === void 0 ? void 0 : (_deletedPolicy$packag2 = deletedPolicy.package) === null || _deletedPolicy$packag2 === void 0 ? void 0 : _deletedPolicy$packag2.name) !== null && _deletedPolicy$packag !== void 0 ? _deletedPolicy$packag : '')) {
      policyIds.push(deletedPolicy.id);
    }
  }
  logger.debug(() => `Looking at deleted integration policies matching package.name of [${packageNames.join(' | ')}] found the following integration policy ids: ${(0, _stringify.stringify)(policyIds)}`);
  if (policyIds.length > 0) {
    const esClient = endpointService.getInternalEsClient();

    // Ensure the index has required mappings
    await (0, _services.ensureActionRequestsIndexIsConfigured)(endpointService);
    try {
      var _updateResponse$failu;
      const updateResponse = await esClient.updateByQuery({
        index: _constants.ENDPOINT_ACTIONS_INDEX,
        conflicts: 'proceed',
        refresh: false,
        ignore_unavailable: true,
        query: {
          bool: {
            filter: {
              terms: {
                'agent.policy.integrationPolicyId': policyIds
              }
            }
          }
        },
        script: {
          lang: 'painless',
          // FIXME:PT REPLACE tag value below with variable defined in PR #224329
          source: `
if (ctx._source.containsKey('tags')) {
  ctx._source.tags.add('INTEGRATION-POLICY-DELETED');
} else {
  ctx._source.tags = ['INTEGRATION-POLICY-DELETED'];
}
`
        }
      }).catch(_utils.catchAndWrapError);
      logger.debug(`Update of response actions that reference integration policy ids [${policyIds.join(' | ')}] done:\n${(0, _stringify.stringify)(updateResponse)}`);
      if ((_updateResponse$failu = updateResponse.failures) !== null && _updateResponse$failu !== void 0 && _updateResponse$failu.length) {
        logger.error(`The following response action updates (as a result of deleted integration policy) failed:${(0, _stringify.stringify)(updateResponse.failures)}`);
      }
    } catch (err) {
      logger.error(`Attempt to updateByQuery() response actions that reference deleted policy ids [${policyIds.join(' | ')}] failed with: ${err.message}`, {
        error: err
      });
    }
  }
};
exports.updateDeletedPolicyResponseActions = updateDeletedPolicyResponseActions;