"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useUpdateRuleByIdCache = exports.useInvalidateFetchRuleByIdQuery = exports.useFetchRuleByIdQuery = void 0;
var _reactQuery = require("@tanstack/react-query");
var _react = require("react");
var _constants = require("../../../../../common/constants");
var _transforms = require("../../../common/transforms");
var _api = require("../api");
var _constants2 = require("./constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const FIND_ONE_RULE_QUERY_KEY = ['GET', _constants.DETECTION_ENGINE_RULES_URL];

/**
 * A wrapper around useQuery provides default values to the underlying query,
 * like query key, abortion signal, and error handler.
 *
 * @param id - rule's id, not rule_id
 * @param options - react-query options
 * @returns useQuery result
 */
const useFetchRuleByIdQuery = (id, options) => {
  return (0, _reactQuery.useQuery)([...FIND_ONE_RULE_QUERY_KEY, id], async ({
    signal
  }) => {
    const response = await (0, _api.fetchRuleById)({
      signal,
      id
    });
    return (0, _transforms.transformInput)(response);
  }, {
    ..._constants2.DEFAULT_QUERY_OPTIONS,
    ...options,
    // Mark this query as immediately stale helps to avoid problems related to filtering.
    // e.g. enabled and disabled state filter require data update which happens at the backend side
    staleTime: 0,
    enabled: !!id
  });
};

/**
 * We should use this hook to invalidate the rules cache. For example, rule
 * mutations that affect rule set size, like creation or deletion, should lead
 * to cache invalidation.
 *
 * @returns A rules cache invalidation callback
 */
exports.useFetchRuleByIdQuery = useFetchRuleByIdQuery;
const useInvalidateFetchRuleByIdQuery = () => {
  const queryClient = (0, _reactQuery.useQueryClient)();
  return (0, _react.useCallback)(() => {
    queryClient.invalidateQueries(FIND_ONE_RULE_QUERY_KEY, {
      refetchType: 'active'
    });
  }, [queryClient]);
};

/**
 * We should use this hook to update the rules cache when modifying a rule.
 * Use it with the new rule data after operations like rule edit.
 *
 * @returns A rules cache update callback
 */
exports.useInvalidateFetchRuleByIdQuery = useInvalidateFetchRuleByIdQuery;
const useUpdateRuleByIdCache = () => {
  const queryClient = (0, _reactQuery.useQueryClient)();
  /**
   * Use this method to update rules data cached by react-query.
   * It is useful when we receive new rules back from a mutation query (bulk edit, etc.);
   * we can merge those rules with the existing cache to avoid an extra roundtrip to re-fetch updated rules.
   */
  return (0, _react.useCallback)(updatedRuleResponse => {
    queryClient.setQueryData([...FIND_ONE_RULE_QUERY_KEY, updatedRuleResponse.id], (0, _transforms.transformInput)(updatedRuleResponse));
  }, [queryClient]);
};
exports.useUpdateRuleByIdCache = useUpdateRuleByIdCache;