"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useNavigateToAppEventHandler = void 0;
var _react = require("react");
var _kibana = require("../../lib/kibana");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Provides an event handlers that can be used with (for example) `onClick` to prevent the
 * event's default behaviour and instead use Kibana's `navigateToApp()` to send user to a
 * different app. Use of `navigateToApp()` prevents a full browser refresh for apps that have
 * been converted to the New Platform.
 *
 * @param appId
 * @param [options]
 *
 * @example
 *
 * const handleOnClick = useNavigateToAppEventHandler('fleet', {path: '/policies'})
 * return <EuiLink onClick={handleOnClick}>See policies</EuiLink>
 */
const useNavigateToAppEventHandler = (appId, options) => {
  const {
    services
  } = (0, _kibana.useKibana)();
  const {
    path,
    state,
    onClick,
    deepLinkId,
    openInNewTab
  } = options || {};
  return (0, _react.useCallback)(ev => {
    try {
      if (onClick) {
        onClick(ev);
      }
    } catch (error) {
      ev.preventDefault();
      throw error;
    }
    if (ev.defaultPrevented) {
      return;
    }
    if (ev.button !== 0) {
      return;
    }
    if (ev.currentTarget instanceof HTMLAnchorElement && ev.currentTarget.target !== '' && ev.currentTarget.target !== '_self') {
      return;
    }
    if (ev.metaKey || ev.altKey || ev.ctrlKey || ev.shiftKey) {
      return;
    }
    ev.preventDefault();
    services.application.navigateToApp(appId, {
      deepLinkId,
      path,
      state,
      openInNewTab
    });
  }, [appId, deepLinkId, onClick, path, services.application, state, openInNewTab]);
};
exports.useNavigateToAppEventHandler = useNavigateToAppEventHandler;