"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.defineRecordViolations = defineRecordViolations;
exports.permissionsPolicyViolationReportSchema = void 0;
var _configSchema = require("@kbn/config-schema");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * # Tracking CSP violations
 *
 * Add the following settings to your `kibana.dev.yml`:
 *
 * ```yml
 * server.customResponseHeaders.Reporting-Endpoints: violations-endpoint="https://localhost:5601/xyz/internal/security/analytics/_record_violations"
 * csp.report_to: [violations-endpoint]
 * ```
 *
 * Note: The endpoint has to be on HTTPS and has to be a fully qualified URL including protocol and
 * hostname, otherwise browsers might not send any reports. When using `server.publicBaseUrl` setting
 * you should use the same origin for the reporting endpoint since Kibana does not support
 * cross-origin content negotiation so browsers would not be able to send any report.
 *
 * # Debugging CSP violations
 *
 * CSP violations are tracked (batched) using event based telemetry.
 *
 * To print telemetry events to the terminal add the following config to your `kibana.dev.yml`:
 *
 * ```yml
 * logging.loggers:
 *   - name: analytics
 *     level: all
 *     appenders: [console]
 * ```
 */

/**
 * Schema that validates CSP violation reports according to W3C spec.
 *
 * https://www.w3.org/TR/CSP3/#reporting
 */
const cspViolationReportSchema = _configSchema.schema.object({
  type: _configSchema.schema.literal('csp-violation'),
  age: _configSchema.schema.maybe(_configSchema.schema.number()),
  url: _configSchema.schema.string(),
  user_agent: _configSchema.schema.maybe(_configSchema.schema.string()),
  body: _configSchema.schema.object({
    documentURL: _configSchema.schema.string(),
    referrer: _configSchema.schema.maybe(_configSchema.schema.string()),
    blockedURL: _configSchema.schema.maybe(_configSchema.schema.string()),
    effectiveDirective: _configSchema.schema.string(),
    originalPolicy: _configSchema.schema.string(),
    sourceFile: _configSchema.schema.maybe(_configSchema.schema.string()),
    sample: _configSchema.schema.maybe(_configSchema.schema.string()),
    disposition: _configSchema.schema.oneOf([_configSchema.schema.literal('enforce'), _configSchema.schema.literal('report')]),
    statusCode: _configSchema.schema.number(),
    lineNumber: _configSchema.schema.maybe(_configSchema.schema.number()),
    columnNumber: _configSchema.schema.maybe(_configSchema.schema.number())
  }, {
    unknowns: 'ignore'
  })
}, {
  unknowns: 'ignore'
});

/**
 * Interface that represents a CSP violation report according to W3C spec.
 */

/**
 * Schema that validates permissions policy violation reports according to W3C spec.
 *
 * https://w3c.github.io/webappsec-permissions-policy/#reporting
 */
const permissionsPolicyViolationReportSchema = exports.permissionsPolicyViolationReportSchema = _configSchema.schema.object({
  type: _configSchema.schema.literal('permissions-policy-violation'),
  age: _configSchema.schema.maybe(_configSchema.schema.number()),
  url: _configSchema.schema.string(),
  user_agent: _configSchema.schema.maybe(_configSchema.schema.string()),
  body: _configSchema.schema.object({
    /**
     * The string identifying the policy-controlled feature whose policy has been violated. This string can be used for grouping and counting related reports.
     * Spec mentions featureId, however the report that is sent from Chrome has policyId. This is to handle both cases.
     */
    policyId: _configSchema.schema.maybe(_configSchema.schema.string()),
    /**
     * The string identifying the policy-controlled feature whose policy has been violated. This string can be used for grouping and counting related reports.
     */
    featureId: _configSchema.schema.maybe(_configSchema.schema.string()),
    /**
     * If known, the file where the violation occured, or null otherwise.
     */
    sourceFile: _configSchema.schema.maybe(_configSchema.schema.string()),
    /**
     * If known, the line number in sourceFile where the violation occured, or null otherwise.
     */
    lineNumber: _configSchema.schema.maybe(_configSchema.schema.number()),
    /**
     * If known, the column number in sourceFile where the violation occured, or null otherwise.
     */
    columnNumber: _configSchema.schema.maybe(_configSchema.schema.number()),
    /**
     * A string indicating whether the violated permissions policy was enforced in this case. disposition will be set to "enforce" if the policy was enforced, or "report" if the violation resulted only in this report being generated (with no further action taken by the user agent in response to the violation).
     */
    disposition: _configSchema.schema.oneOf([_configSchema.schema.literal('enforce'), _configSchema.schema.literal('report')])
  }, {
    unknowns: 'ignore'
  })
}, {
  unknowns: 'ignore'
});

/**
 * Interface that represents a permissions policy violation report according to W3C spec.
 */

/**
 * This endpoint receives reports from the user's browser via the Reporting API when one of our
 * `Content-Security-Policy` or `Permissions-Policy` directives have been violated.
 */
function defineRecordViolations({
  router,
  analyticsService
}) {
  router.post({
    path: '/internal/security/analytics/_record_violations',
    security: {
      authz: {
        enabled: false,
        reason: 'This route is used by browsers to report CSP and Permission Policy violations. These requests are sent without authentication per the browser spec.'
      }
    },
    validate: {
      /**
       * Chrome supports CSP3 spec and sends an array of reports. Safari only sends a single
       * report so catering for both here.
       */
      body: _configSchema.schema.oneOf([_configSchema.schema.arrayOf(_configSchema.schema.oneOf([cspViolationReportSchema, permissionsPolicyViolationReportSchema])), cspViolationReportSchema, permissionsPolicyViolationReportSchema])
    },
    options: {
      /**
       * Browsers will stop sending reports for the duration of the browser session and without
       * further retries once this endpoint has returned a single 403. This would effectively
       * prevent us from capture any reports. To work around this behaviour we optionally
       * authenticate users but silently ignore any reports that have been received from
       * unauthenticated users.
       */
      authRequired: 'optional',
      /**
       * This endpoint is called by the browser in the background so `kbn-xsrf` header is not sent.
       */
      xsrfRequired: false,
      access: 'public',
      body: {
        /**
         * Both `application/reports+json` (CSP3 spec) and `application/csp-report` (Safari) are
         * valid values but Hapi does not parse the request body when `application/csp-report` is
         * specified so enforcing JSON mime-type for this endpoint.
         */
        override: 'application/json'
      }
    }
  }, async (context, request, response) => {
    if (request.auth.isAuthenticated) {
      const reports = Array.isArray(request.body) ? request.body : [request.body];
      const now = Date.now();
      reports.forEach(report => {
        if (report.type === 'csp-violation') {
          var _report$age, _report$user_agent;
          analyticsService.reportCSPViolation({
            created: `${now + ((_report$age = report.age) !== null && _report$age !== void 0 ? _report$age : 0)}`,
            url: report.url,
            user_agent: (_report$user_agent = report.user_agent) !== null && _report$user_agent !== void 0 ? _report$user_agent : getLastHeader(request.headers['user-agent']),
            ...report.body
          });
        } else if (report.type === 'permissions-policy-violation') {
          var _report$age2, _report$user_agent2;
          analyticsService.reportPermissionsPolicyViolation({
            created: `${now + ((_report$age2 = report.age) !== null && _report$age2 !== void 0 ? _report$age2 : 0)}`,
            url: report.url,
            user_agent: (_report$user_agent2 = report.user_agent) !== null && _report$user_agent2 !== void 0 ? _report$user_agent2 : getLastHeader(request.headers['user-agent']),
            ...report.body
          });
        }
      });
    }
    return response.ok();
  });
}
function getLastHeader(header) {
  return Array.isArray(header) ? header[header.length - 1] : header;
}