"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.generateEsqlTool = void 0;
var _zod = require("@kbn/zod");
var _onechatCommon = require("@kbn/onechat-common");
var _onechatGenaiUtils = require("@kbn/onechat-genai-utils");
var _tool_result = require("@kbn/onechat-common/tools/tool_result");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const nlToEsqlToolSchema = _zod.z.object({
  query: _zod.z.string().describe('A natural language query to generate an ES|QL query from.'),
  index: _zod.z.string().optional().describe('(optional) Index to search against. If not provided, will use the index explorer to find the best index to use.'),
  context: _zod.z.string().optional().describe('(optional) Additional context that could be useful to generate the ES|QL query')
});
const generateEsqlTool = () => {
  return {
    id: _onechatCommon.platformCoreTools.generateEsql,
    type: _onechatCommon.ToolType.builtin,
    description: 'Generate an ES|QL query from a natural language query.',
    schema: nlToEsqlToolSchema,
    handler: async ({
      query: nlQuery,
      index,
      context
    }, {
      esClient,
      modelProvider,
      logger,
      events
    }) => {
      const model = await modelProvider.getDefaultModel();
      let selectedTarget = index;
      if (!selectedTarget) {
        const {
          resources: [selectedResource]
        } = await (0, _onechatGenaiUtils.indexExplorer)({
          nlQuery,
          esClient: esClient.asCurrentUser,
          limit: 1,
          model
        });
        selectedTarget = selectedResource.name;
      }
      const esqlResponse = await (0, _onechatGenaiUtils.generateEsql)({
        nlQuery,
        index: selectedTarget,
        additionalContext: context,
        model,
        esClient: esClient.asCurrentUser,
        logger,
        events
      });
      const toolResults = [];
      if (esqlResponse.error) {
        toolResults.push({
          type: _tool_result.ToolResultType.error,
          data: {
            message: esqlResponse.error
          }
        });
      } else {
        if (esqlResponse.query) {
          toolResults.push({
            type: _tool_result.ToolResultType.query,
            data: {
              esql: esqlResponse.query
            }
          });
        }
        if (esqlResponse.answer) {
          toolResults.push({
            type: _tool_result.ToolResultType.other,
            data: {
              answer: esqlResponse.answer
            }
          });
        }
      }
      return {
        results: toolResults
      };
    },
    tags: []
  };
};
exports.generateEsqlTool = generateEsqlTool;