"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useSubscribeToChatEvents = void 0;
var _onechatCommon = require("@kbn/onechat-common");
var _conversation = require("@kbn/onechat-common/chat/conversation");
var _use_conversation_actions = require("../../hooks/use_conversation_actions");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const useSubscribeToChatEvents = ({
  setAgentReasoning,
  setIsResponseLoading,
  isAborted
}) => {
  const conversationActions = (0, _use_conversation_actions.useConversationActions)();
  return events$ => {
    return new Promise((resolve, reject) => {
      events$.subscribe({
        next: event => {
          // chunk received, we append it to the chunk buffer
          if ((0, _onechatCommon.isMessageChunkEvent)(event)) {
            conversationActions.addAssistantMessageChunk({
              messageChunk: event.data.text_chunk
            });
          }
          // full message received, override chunk buffer
          else if ((0, _onechatCommon.isMessageCompleteEvent)(event)) {
            conversationActions.setAssistantMessage({
              assistantMessage: event.data.message_content
            });
          } else if ((0, _onechatCommon.isToolProgressEvent)(event)) {
            conversationActions.setToolCallProgress({
              progress: {
                message: event.data.message
              },
              toolCallId: event.data.tool_call_id
            });
            // Individual tool progression message should also be displayed as reasoning
            setAgentReasoning(event.data.message);
          } else if ((0, _onechatCommon.isReasoningEvent)(event)) {
            conversationActions.addReasoningStep({
              step: (0, _conversation.createReasoningStep)({
                reasoning: event.data.reasoning
              })
            });
            setAgentReasoning(event.data.reasoning);
          } else if ((0, _onechatCommon.isToolCallEvent)(event)) {
            conversationActions.addToolCall({
              step: (0, _conversation.createToolCallStep)({
                params: event.data.params,
                results: [],
                tool_call_id: event.data.tool_call_id,
                tool_id: event.data.tool_id
              })
            });
          } else if ((0, _onechatCommon.isToolResultEvent)(event)) {
            const {
              tool_call_id: toolCallId,
              results
            } = event.data;
            conversationActions.setToolCallResult({
              results,
              toolCallId
            });
          } else if ((0, _onechatCommon.isRoundCompleteEvent)(event)) {
            // Now we have the full response and can stop the loading indicators
            setIsResponseLoading(false);
          } else if ((0, _onechatCommon.isConversationCreatedEvent)(event)) {
            const {
              conversation_id: id,
              title
            } = event.data;
            conversationActions.onConversationCreated({
              conversationId: id,
              title
            });
          }
        },
        complete: () => {
          resolve();
        },
        error: err => {
          // If the request is aborted, we don't want to show an error
          if (isAborted()) {
            setIsResponseLoading(false);
            resolve();
            return;
          }
          reject(err);
        }
      });
    });
  };
};
exports.useSubscribeToChatEvents = useSubscribeToChatEvents;