"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useProductDoc = useProductDoc;
var _react = require("react");
var _reactQuery = require("@tanstack/react-query");
var _constants = require("../constants");
var _use_kibana = require("./use_kibana");
var _use_uninstall_product_doc = require("./use_uninstall_product_doc");
var _use_install_product_doc = require("./use_install_product_doc");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Custom hook to get the status of the product documentation installation.
 * It also provides methods to install and uninstall the product documentation.
 *
 * @param inferenceId - The ID of the inference for which to get the product documentation status.
 * @returns An object containing the status of the product documentation, loading state, and methods to install and uninstall the product documentation.
 */
function useProductDoc(inferenceId) {
  const {
    productDocBase
  } = (0, _use_kibana.useKibana)().services;
  const {
    mutateAsync: installProductDoc,
    isLoading: isInstalling
  } = (0, _use_install_product_doc.useInstallProductDoc)();
  const {
    mutateAsync: uninstallProductDoc,
    isLoading: isUninstalling
  } = (0, _use_uninstall_product_doc.useUninstallProductDoc)();
  const {
    isLoading,
    data,
    refetch,
    isRefetching
  } = (0, _reactQuery.useQuery)({
    networkMode: 'always',
    queryKey: [_constants.REACT_QUERY_KEYS.GET_PRODUCT_DOC_STATUS, inferenceId],
    queryFn: async () => {
      return productDocBase.installation.getStatus({
        inferenceId
      });
    },
    keepPreviousData: false,
    refetchOnWindowFocus: false
  });
  (0, _react.useEffect)(() => {
    refetch();
  }, [inferenceId, refetch]);

  // poll the status if when is installing or uninstalling
  (0, _react.useEffect)(() => {
    if (!((data === null || data === void 0 ? void 0 : data.overall) === 'installing' || (data === null || data === void 0 ? void 0 : data.overall) === 'uninstalling' || isInstalling || isUninstalling)) {
      return;
    }
    const interval = setInterval(refetch, 5000);

    // cleanup the interval if unmount
    return () => {
      clearInterval(interval);
    };
  }, [refetch, data === null || data === void 0 ? void 0 : data.overall, isInstalling, isUninstalling]);
  const status = (0, _react.useMemo)(() => {
    if (!inferenceId || (data === null || data === void 0 ? void 0 : data.inferenceId) !== inferenceId) {
      return undefined;
    }
    if (isInstalling) {
      return 'installing';
    }
    if (isUninstalling) {
      return 'uninstalling';
    }
    return data === null || data === void 0 ? void 0 : data.overall;
  }, [inferenceId, isInstalling, isUninstalling, data]);
  return {
    status,
    isLoading: isLoading || isRefetching || isInstalling || isUninstalling,
    installProductDoc,
    uninstallProductDoc
  };
}