"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.setupConsoleErrorsProvider = void 0;
var _constants = require("./constants");
var _monaco_imports = require("../../monaco_imports");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/*
 * This setup function runs when the Console language is registered into the Monaco editor.
 * It adds a listener that is attached to the editor input when the Monaco editor is used
 * with the Console language.
 * The Console parser that runs in a web worker analyzes the editor input when it changes and
 * if any errors are found, they are added as "error markers" to the Monaco editor.
 */
const setupConsoleErrorsProvider = workerProxyService => {
  const updateErrorMarkers = async model => {
    if (model.isDisposed()) {
      return;
    }
    const parserResult = await workerProxyService.getParserResult(model.uri);
    if (!parserResult) {
      return;
    }
    const {
      errors
    } = parserResult;
    _monaco_imports.monaco.editor.setModelMarkers(model, _constants.CONSOLE_LANG_ID, errors.map(({
      offset,
      text
    }) => {
      const {
        column,
        lineNumber
      } = model.getPositionAt(offset);
      return {
        startLineNumber: lineNumber,
        startColumn: column,
        endLineNumber: lineNumber,
        endColumn: column,
        message: text,
        severity: _monaco_imports.monaco.MarkerSeverity.Error
      };
    }));
  };
  const onModelAdd = model => {
    if (model.getLanguageId() !== _constants.CONSOLE_LANG_ID) {
      return;
    }
    const {
      dispose
    } = model.onDidChangeContent(async () => {
      await updateErrorMarkers(model);
    });
    model.onWillDispose(() => {
      dispose();
    });
    updateErrorMarkers(model);
  };
  _monaco_imports.monaco.editor.onDidCreateModel(onModelAdd);
};
exports.setupConsoleErrorsProvider = setupConsoleErrorsProvider;