"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useCreateAnalyticsForm = void 0;
var _react = require("react");
var _i18n = require("@kbn/i18n");
var _mlErrorUtils = require("@kbn/ml-error-utils");
var _kibana = require("../../../../../contexts/kibana");
var _common = require("../../../../common");
var _action_clone = require("../../components/action_clone");
var _actions = require("./actions");
var _reducer = require("./reducer");
var _state = require("./state");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const useCreateAnalyticsForm = () => {
  const {
    services: {
      data: {
        dataViews
      }
    }
  } = (0, _kibana.useMlKibana)();
  const mlApi = (0, _kibana.useMlApi)();
  const [state, dispatch] = (0, _react.useReducer)(_reducer.reducer, (0, _state.getInitialState)());
  const {
    refresh
  } = (0, _common.useRefreshAnalyticsList)();
  const {
    form,
    jobConfig,
    isAdvancedEditorEnabled
  } = state;
  const {
    createDataView,
    jobId
  } = form;
  const addRequestMessage = requestMessage => dispatch({
    type: _actions.ACTION.ADD_REQUEST_MESSAGE,
    requestMessage
  });
  const closeModal = () => dispatch({
    type: _actions.ACTION.CLOSE_MODAL
  });
  const resetAdvancedEditorMessages = () => dispatch({
    type: _actions.ACTION.RESET_ADVANCED_EDITOR_MESSAGES
  });
  const setAdvancedEditorRawString = advancedEditorRawString => dispatch({
    type: _actions.ACTION.SET_ADVANCED_EDITOR_RAW_STRING,
    advancedEditorRawString
  });
  const setDataViewTitles = payload => dispatch({
    type: _actions.ACTION.SET_DATA_VIEW_TITLES,
    payload
  });
  const setIsJobCreated = isJobCreated => dispatch({
    type: _actions.ACTION.SET_IS_JOB_CREATED,
    isJobCreated
  });
  const setIsJobStarted = isJobStarted => {
    dispatch({
      type: _actions.ACTION.SET_IS_JOB_STARTED,
      isJobStarted
    });
  };
  const resetRequestMessages = () => dispatch({
    type: _actions.ACTION.RESET_REQUEST_MESSAGES
  });
  const resetForm = () => dispatch({
    type: _actions.ACTION.RESET_FORM
  });
  const createAnalyticsJob = async () => {
    resetRequestMessages();
    const analyticsJobConfig = isAdvancedEditorEnabled ? jobConfig : (0, _state.getJobConfigFromFormState)(form);
    const errorMessage = _i18n.i18n.translate('xpack.ml.dataframe.analytics.create.errorCreatingDataFrameAnalyticsJob', {
      defaultMessage: 'An error occurred creating the data frame analytics job:'
    });
    try {
      const creationResp = await mlApi.dataFrameAnalytics.createDataFrameAnalytics(jobId, analyticsJobConfig, createDataView, form.timeFieldName);
      addRequestMessage({
        message: _i18n.i18n.translate('xpack.ml.dataframe.stepCreateForm.createDataFrameAnalyticsSuccessMessage', {
          defaultMessage: 'Request to create data frame analytics {jobId} acknowledged.',
          values: {
            jobId
          }
        })
      });
      if (creationResp.dataFrameAnalyticsJobsCreated.length && creationResp.dataFrameAnalyticsJobsErrors.length === 0) {
        setIsJobCreated(true);
        refresh();
        return true;
      } else if (creationResp.dataFrameAnalyticsJobsErrors.length) {
        addRequestMessage({
          error: (0, _mlErrorUtils.extractErrorProperties)(creationResp.dataFrameAnalyticsJobsErrors[0].error).message,
          message: errorMessage
        });
        return false;
      }
    } catch (e) {
      addRequestMessage({
        error: (0, _mlErrorUtils.extractErrorMessage)(e),
        message: errorMessage
      });
      return false;
    }
    return false;
  };
  const prepareFormValidation = async () => {
    try {
      // Set the existing data view names.
      const dataViewsMap = {};
      const savedObjects = (await dataViews.getCache()) || [];
      savedObjects.forEach(obj => {
        var _obj$attributes;
        const title = obj === null || obj === void 0 ? void 0 : (_obj$attributes = obj.attributes) === null || _obj$attributes === void 0 ? void 0 : _obj$attributes.title;
        if (title !== undefined) {
          const id = (obj === null || obj === void 0 ? void 0 : obj.id) || '';
          dataViewsMap[title] = {
            label: title,
            value: id
          };
        }
      });
      setDataViewTitles({
        dataViewsMap
      });
    } catch (e) {
      addRequestMessage({
        error: (0, _mlErrorUtils.extractErrorMessage)(e),
        message: _i18n.i18n.translate('xpack.ml.dataframe.analytics.create.errorGettingDataViewNames', {
          defaultMessage: 'An error occurred getting the existing data view names:'
        })
      });
    }
  };
  const initiateWizard = async () => {
    await dataViews.clearCache();
    await prepareFormValidation();
  };
  const startAnalyticsJob = async () => {
    try {
      const response = await mlApi.dataFrameAnalytics.startDataFrameAnalytics(jobId);
      if (response.acknowledged !== true) {
        throw new Error(response);
      }
      addRequestMessage({
        message: _i18n.i18n.translate('xpack.ml.dataframe.analytics.create.startDataFrameAnalyticsSuccessMessage', {
          defaultMessage: 'Request to start data frame analytics {jobId} acknowledged.',
          values: {
            jobId
          }
        })
      });
      setIsJobStarted(true);
      refresh();
    } catch (e) {
      addRequestMessage({
        error: (0, _mlErrorUtils.extractErrorMessage)(e),
        message: _i18n.i18n.translate('xpack.ml.dataframe.analytics.create.errorStartingDataFrameAnalyticsJob', {
          defaultMessage: 'An error occurred starting the data frame analytics job:'
        })
      });
    }
  };
  const setJobConfig = payload => {
    dispatch({
      type: _actions.ACTION.SET_JOB_CONFIG,
      payload
    });
  };
  const setFormState = payload => {
    dispatch({
      type: _actions.ACTION.SET_FORM_STATE,
      payload
    });
  };
  const switchToAdvancedEditor = () => {
    dispatch({
      type: _actions.ACTION.SWITCH_TO_ADVANCED_EDITOR
    });
  };
  const switchToForm = () => {
    dispatch({
      type: _actions.ACTION.SWITCH_TO_FORM
    });
  };
  const setEstimatedModelMemoryLimit = value => {
    dispatch({
      type: _actions.ACTION.SET_ESTIMATED_MODEL_MEMORY_LIMIT,
      value: value !== null && value !== void 0 ? value : ''
    });
  };
  const setJobClone = async cloneJob => {
    resetForm();
    const config = (0, _action_clone.extractCloningConfig)(cloneJob);
    if ((0, _action_clone.isAdvancedConfig)(config)) {
      setFormState((0, _state.getFormStateFromJobConfig)(config));
      switchToAdvancedEditor();
    } else {
      setFormState((0, _state.getFormStateFromJobConfig)(config));
      setEstimatedModelMemoryLimit(config.model_memory_limit);
    }
    dispatch({
      type: _actions.ACTION.SET_JOB_CLONE,
      cloneJob
    });
  };
  const actions = {
    closeModal,
    createAnalyticsJob,
    initiateWizard,
    resetAdvancedEditorMessages,
    setAdvancedEditorRawString,
    setFormState,
    setJobConfig,
    startAnalyticsJob,
    switchToAdvancedEditor,
    switchToForm,
    setEstimatedModelMemoryLimit,
    setJobClone
  };
  return {
    state,
    actions
  };
};
exports.useCreateAnalyticsForm = useCreateAnalyticsForm;