"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.VERSION = exports.TYPE = exports.DeleteUnenrolledAgentsTask = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _server = require("@kbn/core/server");
var _task = require("@kbn/task-manager-plugin/server/task");
var _elasticsearch = require("@elastic/elasticsearch");
var _constants = require("../../common/constants");
var _services = require("../services");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const TYPE = exports.TYPE = 'fleet:delete-unenrolled-agents-task';
const VERSION = exports.VERSION = '1.0.0';
const TITLE = 'Fleet Delete Unenrolled Agents Task';
const SCOPE = ['fleet'];
const INTERVAL = '1h';
const TIMEOUT = '1m';
class DeleteUnenrolledAgentsTask {
  constructor(setupContract) {
    (0, _defineProperty2.default)(this, "logger", void 0);
    (0, _defineProperty2.default)(this, "wasStarted", false);
    (0, _defineProperty2.default)(this, "start", async ({
      taskManager
    }) => {
      if (!taskManager) {
        this.logger.error('[DeleteUnenrolledAgentsTask] Missing required service during start');
        return;
      }
      this.wasStarted = true;
      this.logger.info(`[DeleteUnenrolledAgentsTask] Started with interval of [${INTERVAL}]`);
      try {
        await taskManager.ensureScheduled({
          id: this.taskId,
          taskType: TYPE,
          scope: SCOPE,
          schedule: {
            interval: INTERVAL
          },
          state: {},
          params: {
            version: VERSION
          }
        });
      } catch (e) {
        this.logger.error(`Error scheduling task DeleteUnenrolledAgentsTask, error: ${e.message}`, e);
      }
    });
    (0, _defineProperty2.default)(this, "runTask", async ({
      taskInstance,
      core,
      abortController
    }) => {
      if (!this.wasStarted) {
        this.logger.debug('[DeleteUnenrolledAgentsTask] runTask Aborted. Task not started yet');
        return;
      }
      // Check that this task is current
      if (taskInstance.id !== this.taskId) {
        this.logger.debug(`[DeleteUnenrolledAgentsTask] Outdated task version: Got [${taskInstance.id}] from task instance. Current version is [${this.taskId}]`);
        return (0, _task.getDeleteTaskRunResult)();
      }
      this.logger.info(`[runTask()] started`);
      const [coreStart] = await core.getStartServices();
      const esClient = coreStart.elasticsearch.client.asInternalUser;
      const soClient = new _server.SavedObjectsClient(coreStart.savedObjects.createInternalRepository());
      try {
        if (!(await this.isDeleteUnenrolledAgentsEnabled(soClient))) {
          this.logger.debug('[DeleteUnenrolledAgentsTask] Delete unenrolled agents flag is disabled, returning.');
          this.endRun('Delete unenrolled agents is disabled');
          return;
        }
        await this.deleteUnenrolledAgents({
          esClient,
          abortController
        });
        this.endRun('success');
      } catch (err) {
        if (err instanceof _elasticsearch.errors.RequestAbortedError) {
          this.logger.warn(`[DeleteUnenrolledAgentsTask] request aborted due to timeout: ${err}`);
          this.endRun();
          return;
        }
        this.logger.error(`[DeleteUnenrolledAgentsTask] error: ${err}`);
        this.endRun('error');
      }
    });
    const {
      core: _core,
      taskManager: _taskManager,
      logFactory
    } = setupContract;
    this.logger = logFactory.get(this.taskId);
    _taskManager.registerTaskDefinitions({
      [TYPE]: {
        title: TITLE,
        timeout: TIMEOUT,
        createTaskRunner: ({
          taskInstance,
          abortController
        }) => {
          return {
            run: async () => {
              return this.runTask({
                taskInstance,
                core: _core,
                abortController
              });
            },
            cancel: async () => {}
          };
        }
      }
    });
  }
  get taskId() {
    return `${TYPE}:${VERSION}`;
  }
  endRun(msg = '') {
    this.logger.info(`[DeleteUnenrolledAgentsTask] runTask ended${msg ? ': ' + msg : ''}`);
  }
  async deleteUnenrolledAgents({
    esClient,
    abortController
  }) {
    this.logger.debug(`[DeleteUnenrolledAgentsTask] Fetching unenrolled agents`);
    const response = await esClient.deleteByQuery({
      index: _constants.AGENTS_INDEX,
      query: {
        bool: {
          filter: [{
            term: {
              active: false
            }
          }]
        }
      }
    }, {
      signal: abortController.signal
    });
    this.logger.debug(`[DeleteUnenrolledAgentsTask] Executed deletion of ${response.deleted} unenrolled agents`);
  }
  async isDeleteUnenrolledAgentsEnabled(soClient) {
    var _settings$delete_unen, _settings$delete_unen2;
    const settings = await _services.settingsService.getSettingsOrUndefined(soClient);
    return (_settings$delete_unen = settings === null || settings === void 0 ? void 0 : (_settings$delete_unen2 = settings.delete_unenrolled_agents) === null || _settings$delete_unen2 === void 0 ? void 0 : _settings$delete_unen2.enabled) !== null && _settings$delete_unen !== void 0 ? _settings$delete_unen : false;
  }
}
exports.DeleteUnenrolledAgentsTask = DeleteUnenrolledAgentsTask;