"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useQuickStartCreateForm = void 0;
var _react = require("react");
var _i18n = require("@kbn/i18n");
var _errors = require("../../../../../../common/errors");
var _agent_policies_helpers = require("../../../../../../common/services/agent_policies_helpers");
var _hooks = require("../../../hooks");
var _use_service_token = require("../../../hooks/use_service_token");
var _use_select_fleet_server_policy = require("./use_select_fleet_server_policy");
var _use_fleet_server_host = require("./use_fleet_server_host");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function getQuickStartFleetServerPolicyFields(spaceId) {
  return {
    id: (0, _agent_policies_helpers.getDefaultFleetServerpolicyId)(spaceId),
    name: 'Fleet Server Policy',
    description: 'Fleet Server policy generated by Kibana',
    namespace: 'default',
    has_fleet_server: true,
    monitoring_enabled: ['logs', 'metrics'],
    is_default_fleet_server: true
  };
}
/**
 * Provides a unified interface that combines the following operations:
 * 1. Setting a Fleet Server host in Fleet's settings
 * 2. Creating an agent policy that contains the `fleet_server` integration
 * 3. Generating a service token used by Fleet Server
 */
const useQuickStartCreateForm = () => {
  const [status, setStatus] = (0, _react.useState)('initial');
  const [error, setError] = (0, _react.useState)();
  const {
    fleetServerHosts,
    fleetServerHost,
    isFleetServerHostSubmitted,
    handleSubmitForm,
    error: fleetServerError,
    setFleetServerHost,
    inputs
  } = (0, _use_fleet_server_host.useFleetServerHost)();
  const {
    spaceId
  } = (0, _hooks.useFleetStatus)();

  // When a validation error is surfaced from the Fleet Server host form, we want to treat it
  // the same way we do errors from the service token or policy creation steps
  (0, _react.useEffect)(() => {
    setStatus('error');
    setError(fleetServerError);
  }, [fleetServerError]);
  const {
    notifications
  } = (0, _hooks.useStartServices)();
  const {
    fleetServerPolicyId,
    setFleetServerPolicyId
  } = (0, _use_select_fleet_server_policy.useSelectFleetServerPolicy)();
  const {
    serviceToken,
    generateServiceToken
  } = (0, _use_service_token.useServiceToken)();
  const quickStartFleetServerPolicyFields = (0, _react.useMemo)(() => getQuickStartFleetServerPolicyFields(spaceId), [spaceId]);
  const submit = (0, _react.useCallback)(async () => {
    try {
      if (!fleetServerHost || fleetServerHost) {
        var _existingPolicy$data;
        setStatus('loading');
        if (!fleetServerHost) {
          const res = await handleSubmitForm();
          if (res) {
            setFleetServerHost(res);
          } else {
            setStatus('initial');
            return;
          }
        }
        await generateServiceToken();
        const existingPolicy = await (0, _hooks.sendGetOneAgentPolicy)(quickStartFleetServerPolicyFields.id);

        // Don't attempt to create the policy if it's already been created in a previous quick start flow
        if ((_existingPolicy$data = existingPolicy.data) !== null && _existingPolicy$data !== void 0 && _existingPolicy$data.item) {
          var _existingPolicy$data2;
          setFleetServerPolicyId((_existingPolicy$data2 = existingPolicy.data) === null || _existingPolicy$data2 === void 0 ? void 0 : _existingPolicy$data2.item.id);
        } else {
          const createPolicyResponse = await (0, _hooks.sendCreateAgentPolicyForRq)(quickStartFleetServerPolicyFields, {
            withSysMonitoring: true
          });
          setFleetServerPolicyId(createPolicyResponse.item.id);
        }
        setStatus('success');
      }
    } catch (err) {
      var _err$attributes;
      if ((err === null || err === void 0 ? void 0 : (_err$attributes = err.attributes) === null || _err$attributes === void 0 ? void 0 : _err$attributes.type) === _errors.OutputInvalidError.name) {
        notifications.toasts.addError(err, {
          title: _i18n.i18n.translate('xpack.fleet.fleetServerSetup.errorCreatingFleetServerPolicyTitle', {
            defaultMessage: 'Error creating a Fleet Server policy'
          }),
          toastMessage: _i18n.i18n.translate('xpack.fleet.fleetServerSetup.errorCreatingFleetServerPolicyMessage', {
            defaultMessage: 'Fleet Server policy creation failed as your default output is not an elasticsearch output. Use the advanced section to use an elasticsearch output to create that policy.'
          })
        });
      } else {
        notifications.toasts.addError(err, {
          title: _i18n.i18n.translate('xpack.fleet.fleetServerSetup.errorAddingFleetServerHostTitle', {
            defaultMessage: 'Error adding Fleet Server host'
          })
        });
      }
      setStatus('error');
      setError(err.message);
    }
  }, [fleetServerHost, handleSubmitForm, setFleetServerHost, generateServiceToken, setFleetServerPolicyId, notifications.toasts, quickStartFleetServerPolicyFields]);
  return {
    status,
    error,
    submit,
    fleetServerPolicyId,
    fleetServerHosts,
    fleetServerHost,
    setFleetServerHost,
    isFleetServerHostSubmitted,
    serviceToken,
    inputs
  };
};
exports.useQuickStartCreateForm = useQuickStartCreateForm;