"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.initializeDataControlManager = exports.defaultDataControlComparators = void 0;
var _rxjs = require("rxjs");
var _common = require("@kbn/data-views-plugin/common");
var _i18n = require("@kbn/i18n");
var _state_manager = require("@kbn/presentation-publishing/state_manager");
var _kibana_services = require("../../services/kibana_services");
var _default_control_manager = require("../default_control_manager");
var _open_data_control_editor = require("./open_data_control_editor");
var _reference_name_utils = require("./reference_name_utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const defaultDataControlComparators = exports.defaultDataControlComparators = {
  ..._default_control_manager.defaultControlComparators,
  title: 'referenceEquality',
  dataViewId: 'referenceEquality',
  fieldName: 'referenceEquality'
};
const initializeDataControlManager = (controlId, controlType, state, getEditorState, setEditorState, controlGroupApi) => {
  const dataControlManager = (0, _state_manager.initializeStateManager)(state, {
    ..._default_control_manager.defaultControlDefaultValues,
    dataViewId: '',
    fieldName: '',
    title: undefined
  }, defaultDataControlComparators);
  const blockingError$ = new _rxjs.BehaviorSubject(undefined);
  function setBlockingError(error) {
    blockingError$.next(error);
  }
  const dataLoading$ = new _rxjs.BehaviorSubject(false);
  function setDataLoading(loading) {
    dataLoading$.next(loading);
  }
  const defaultTitle$ = new _rxjs.BehaviorSubject(undefined);
  const dataViews$ = new _rxjs.BehaviorSubject(undefined);
  const filters$ = new _rxjs.BehaviorSubject(undefined);
  const filtersReady$ = new _rxjs.BehaviorSubject(false);
  const field$ = new _rxjs.BehaviorSubject(undefined);
  const fieldFormatter = new _rxjs.BehaviorSubject(toFormat => String(toFormat));
  const dataViewIdSubscription = dataControlManager.api.dataViewId$.pipe((0, _rxjs.tap)(() => {
    filtersReady$.next(false);
    if (blockingError$.value) {
      setBlockingError(undefined);
    }
  }), (0, _rxjs.switchMap)(async currentDataViewId => {
    let dataView;
    try {
      dataView = await _kibana_services.dataViewsService.get(currentDataViewId);
      return {
        dataView
      };
    } catch (error) {
      return {
        error
      };
    }
  })).subscribe(({
    dataView,
    error
  }) => {
    if (error) {
      setBlockingError(error);
    }
    dataViews$.next(dataView ? [dataView] : undefined);
  });
  const fieldNameSubscription = (0, _rxjs.combineLatest)([dataViews$, dataControlManager.api.fieldName$]).pipe((0, _rxjs.tap)(() => {
    filtersReady$.next(false);
  })).subscribe(([nextDataViews, nextFieldName]) => {
    const dataView = nextDataViews ? nextDataViews.find(({
      id
    }) => dataControlManager.api.dataViewId$.value === id) : undefined;
    if (!dataView) {
      return;
    }
    const field = dataView.getFieldByName(nextFieldName);
    if (!field) {
      setBlockingError(new Error(_i18n.i18n.translate('controls.dataControl.fieldNotFound', {
        defaultMessage: 'Could not locate field: {fieldName}',
        values: {
          fieldName: nextFieldName
        }
      })));
    } else if (blockingError$.value) {
      setBlockingError(undefined);
    }
    field$.next(field);
    defaultTitle$.next(field ? field.displayName || field.name : nextFieldName);
    const spec = field === null || field === void 0 ? void 0 : field.toSpec();
    if (spec) {
      fieldFormatter.next(dataView.getFormatterForField(spec).getConverterFor('text'));
    }
  });
  const onEdit = async () => {
    const initialState = {
      ...dataControlManager.getLatestState(),
      ...getEditorState()
    };

    // open the editor to get the new state
    (0, _open_data_control_editor.openDataControlEditor)({
      onSave: ({
        type: newType,
        state: newState
      }) => {
        if (newType === controlType) {
          dataControlManager.reinitializeState(newState);
          setEditorState(newState);
        } else {
          // replace the control with a new one of the updated type
          controlGroupApi.replacePanel(controlId, {
            panelType: newType,
            serializedState: {
              rawState: newState
            }
          });
        }
      },
      initialState: {
        ...initialState
      },
      controlType,
      controlId,
      initialDefaultPanelTitle: defaultTitle$.getValue(),
      controlGroupApi
    });
  };
  const filtersReadySubscription = filters$.pipe((0, _rxjs.skip)(1)).subscribe(() => {
    // Set filtersReady$.next(true); in filters$ subscription instead of setOutputFilter
    // to avoid signaling filters ready until after filters have been emitted
    // to avoid timing issues
    filtersReady$.next(true);
  });
  return {
    api: {
      ...dataControlManager.api,
      dataLoading$,
      blockingError$,
      setBlockingError,
      setDataLoading,
      defaultTitle$,
      dataViews$,
      field$,
      fieldFormatter,
      onEdit,
      filters$,
      isEditingEnabled: () => true,
      untilFiltersReady: async () => {
        return new Promise(resolve => {
          (0, _rxjs.combineLatest)([blockingError$, filtersReady$]).pipe((0, _rxjs.first)(([blockingError, filtersReady]) => filtersReady || blockingError !== undefined)).subscribe(() => {
            resolve();
          });
        });
      }
    },
    cleanup: () => {
      dataViewIdSubscription.unsubscribe();
      fieldNameSubscription.unsubscribe();
      filtersReadySubscription.unsubscribe();
    },
    internalApi: {
      extractReferences: referenceNameSuffix => {
        return [{
          name: (0, _reference_name_utils.getReferenceName)(controlId, referenceNameSuffix),
          type: _common.DATA_VIEW_SAVED_OBJECT_TYPE,
          id: dataControlManager.api.dataViewId$.getValue()
        }];
      },
      onSelectionChange: () => {
        filtersReady$.next(false);
      },
      setOutputFilter: newFilter => {
        filters$.next(newFilter ? [newFilter] : undefined);
      }
    },
    anyStateChange$: dataControlManager.anyStateChange$,
    getLatestState: dataControlManager.getLatestState,
    reinitializeState: dataControlManager.reinitializeState
  };
};
exports.initializeDataControlManager = initializeDataControlManager;