"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.bulkEditFieldsToExcludeFromRevisionUpdates = void 0;
exports.bulkEditRules = bulkEditRules;
var _boom = _interopRequireDefault(require("@hapi/boom"));
var _lodash = require("lodash");
var _bulk_edit = require("../../../../rules_client/common/bulk_edit");
var _validate_authorize_system_actions = require("../../../../lib/validate_authorize_system_actions");
var _saved_objects = require("../../../../saved_objects");
var _authorization = require("../../../../authorization");
var _common = require("../../../../../common");
var _audit_events = require("../../../../rules_client/common/audit_events");
var _common2 = require("../../../../rules_client/common");
var _lib = require("../../../../rules_client/lib");
var _constants = require("../../constants");
var _schemas = require("../../schemas");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const bulkEditFieldsToExcludeFromRevisionUpdates = exports.bulkEditFieldsToExcludeFromRevisionUpdates = new Set(['snoozeSchedule', 'apiKey']);
async function bulkEditRules(context, options) {
  const shouldInvalidateApiKeys = true;
  const auditAction = _audit_events.RuleAuditAction.BULK_EDIT;
  const requiredAuthOperation = _authorization.WriteOperations.BulkEdit;
  const result = await (0, _bulk_edit.bulkEditRules)(context, {
    ...options,
    name: `rulesClient.bulkEditRules('operations=${JSON.stringify(options.operations)}, paramsModifier=${options.paramsModifier ? '[Function]' : undefined}', shouldIncrementRevision=${options.shouldIncrementRevision ? '[Function]' : undefined}')`,
    auditAction,
    requiredAuthOperation,
    shouldInvalidateApiKeys,
    shouldValidateSchedule: options.operations.some(operation => operation.field === 'schedule'),
    updateFn: opts => updateRuleAttributesAndParamsInMemory({
      ...opts,
      context,
      shouldInvalidateApiKeys,
      operations: options.operations,
      paramsModifier: options.paramsModifier,
      shouldIncrementRevision: options.shouldIncrementRevision
    })
  });
  await bulkUpdateTaskSchedules(context, options.operations, result.rules);
  return result;
}
async function bulkUpdateTaskSchedules(context, operations, updatedRules) {
  const scheduleOperation = operations.find(operation => operation.field === 'schedule');
  if (!(scheduleOperation !== null && scheduleOperation !== void 0 && scheduleOperation.value)) {
    return;
  }
  const taskIds = updatedRules.reduce((acc, rule) => {
    if (rule.scheduledTaskId) {
      acc.push(rule.scheduledTaskId);
    }
    return acc;
  }, []);
  try {
    await context.taskManager.bulkUpdateSchedules(taskIds, scheduleOperation.value);
    context.logger.debug(`Successfully updated schedules for underlying tasks: ${taskIds.join(', ')}`);
  } catch (error) {
    context.logger.error(`Failure to update schedules for underlying tasks: ${taskIds.join(', ')}. TaskManager bulkUpdateSchedules failed with Error: ${error.message}`);
  }
}
async function updateRuleAttributesAndParamsInMemory({
  context,
  operations,
  paramsModifier,
  rule,
  apiKeysMap,
  rules,
  skipped,
  errors,
  username,
  shouldInvalidateApiKeys,
  shouldIncrementRevision = () => true
}) {
  try {
    await ensureAuthorizationForBulkUpdate(context, operations, rule);
    await (0, _bulk_edit.updateRuleInMemory)(context, {
      rule,
      apiKeysMap,
      rules,
      skipped,
      errors,
      username,
      paramsModifier,
      shouldInvalidateApiKeys,
      shouldIncrementRevision,
      updateAttributesFn: async ({
        domainRule,
        ruleActions,
        ruleType
      }) => {
        const result = await getUpdatedAttributesFromOperations({
          context,
          operations,
          rule: domainRule,
          ruleActions,
          ruleType
        });

        // validate the updated schedule interval
        validateScheduleInterval(context, result.rule.schedule.interval, ruleType.id, domainRule.id);
        return result;
      }
    });
  } catch (error) {
    var _rule$attributes, _context$auditLogger, _rule$attributes2;
    errors.push({
      message: error.message,
      rule: {
        id: rule.id,
        name: (_rule$attributes = rule.attributes) === null || _rule$attributes === void 0 ? void 0 : _rule$attributes.name
      }
    });
    (_context$auditLogger = context.auditLogger) === null || _context$auditLogger === void 0 ? void 0 : _context$auditLogger.log((0, _audit_events.ruleAuditEvent)({
      action: _audit_events.RuleAuditAction.BULK_EDIT,
      savedObject: {
        type: _saved_objects.RULE_SAVED_OBJECT_TYPE,
        id: rule.id,
        name: (_rule$attributes2 = rule.attributes) === null || _rule$attributes2 === void 0 ? void 0 : _rule$attributes2.name
      },
      error
    }));
  }
}
async function ensureAuthorizationForBulkUpdate(context, operations, rule) {
  if (rule.attributes.actions.length === 0) {
    return;
  }
  for (const operation of operations) {
    const {
      field
    } = operation;
    if (field === 'snoozeSchedule' || field === 'apiKey') {
      try {
        await context.actionsAuthorization.ensureAuthorized({
          operation: 'execute'
        });
        break;
      } catch (error) {
        throw Error(`Rule not authorized for bulk ${field} update - ${error.message}`);
      }
    }
  }
}
async function getUpdatedAttributesFromOperations({
  context,
  operations,
  rule,
  ruleActions,
  ruleType
}) {
  const actionsClient = await context.getActionsClient();
  let updatedRule = (0, _lodash.cloneDeep)(rule);
  let updatedRuleActions = ruleActions;
  let hasUpdateApiKeyOperation = false;
  let isAttributesUpdateSkipped = true;
  for (const operation of operations) {
    // Check if the update should be skipped for the current action.
    // If it should, save the skip reasons in attributesUpdateSkipReasons
    // and continue to the next operation before without
    // the `isAttributesUpdateSkipped` flag to false.
    switch (operation.field) {
      case 'actions':
        {
          const systemActions = operation.value.filter(action => actionsClient.isSystemAction(action.id));
          const actions = operation.value.filter(action => !actionsClient.isSystemAction(action.id));
          systemActions.forEach(systemAction => {
            try {
              _schemas.systemActionRequestSchema.validate(systemAction);
            } catch (error) {
              throw _boom.default.badRequest(`Error validating bulk edit rules operations - ${error.message}`);
            }
          });
          actions.forEach(action => {
            try {
              _schemas.actionRequestSchema.validate(action);
            } catch (error) {
              throw _boom.default.badRequest(`Error validating bulk edit rules operations - ${error.message}`);
            }
          });
          const {
            actions: genActions,
            systemActions: genSystemActions
          } = await (0, _lib.addGeneratedActionValues)(actions, systemActions, context);
          const updatedOperation = {
            ...operation,
            value: [...genActions, ...genSystemActions]
          };
          await (0, _validate_authorize_system_actions.validateAndAuthorizeSystemActions)({
            actionsClient,
            actionsAuthorization: context.actionsAuthorization,
            connectorAdapterRegistry: context.connectorAdapterRegistry,
            systemActions: genSystemActions,
            rule: {
              consumer: updatedRule.consumer,
              producer: ruleType.producer
            }
          });
          try {
            await (0, _lib.validateActions)(context, ruleType, {
              ...updatedRule,
              actions: genActions,
              systemActions: genSystemActions
            });
          } catch (e) {
            // If validateActions fails on the first attempt, it may be because of legacy rule-level frequency params
            updatedRule = await attemptToMigrateLegacyFrequency(context, operation.field, genActions, updatedRule, ruleType);
          }
          const {
            modifiedAttributes,
            isAttributeModified
          } = (0, _common2.applyBulkEditOperation)(updatedOperation, {
            actions: updatedRuleActions
          });
          if (isAttributeModified) {
            updatedRuleActions = modifiedAttributes.actions;
            isAttributesUpdateSkipped = false;
          }
          break;
        }
      case 'snoozeSchedule':
        {
          if (operation.operation === 'set') {
            const snoozeAttributes = (0, _common2.getBulkSnooze)(updatedRule, operation.value);
            try {
              (0, _common2.verifySnoozeScheduleLimit)(snoozeAttributes.snoozeSchedule);
            } catch (error) {
              throw Error(`Error updating rule: could not add snooze - ${error.message}`);
            }
            updatedRule = {
              ...updatedRule,
              muteAll: snoozeAttributes.muteAll,
              snoozeSchedule: snoozeAttributes.snoozeSchedule
            };
          }
          if (operation.operation === 'delete') {
            const idsToDelete = operation.value && [...operation.value];
            if ((idsToDelete === null || idsToDelete === void 0 ? void 0 : idsToDelete.length) === 0) {
              var _updatedRule$snoozeSc;
              (_updatedRule$snoozeSc = updatedRule.snoozeSchedule) === null || _updatedRule$snoozeSc === void 0 ? void 0 : _updatedRule$snoozeSc.forEach(schedule => {
                if (schedule.id) {
                  idsToDelete.push(schedule.id);
                }
              });
            }
            const snoozeAttributes = (0, _common2.getBulkUnsnooze)(updatedRule, idsToDelete);
            updatedRule = {
              ...updatedRule,
              muteAll: snoozeAttributes.muteAll,
              snoozeSchedule: snoozeAttributes.snoozeSchedule
            };
          }
          isAttributesUpdateSkipped = false;
          break;
        }
      case 'apiKey':
        {
          hasUpdateApiKeyOperation = true;
          isAttributesUpdateSkipped = false;
          break;
        }
      default:
        {
          if (operation.field === 'schedule') {
            const defaultActions = updatedRule.actions.filter(action => !actionsClient.isSystemAction(action.id));
            validateScheduleOperation(operation.value, defaultActions, rule.id);
          }
          const {
            modifiedAttributes,
            isAttributeModified
          } = (0, _common2.applyBulkEditOperation)(operation, updatedRule);
          if (isAttributeModified) {
            updatedRule = {
              ...updatedRule,
              ...modifiedAttributes
            };
            isAttributesUpdateSkipped = false;
          }
        }
    }
    // Only increment revision if update wasn't skipped and `operation.field` should result in a revision increment
    if (!isAttributesUpdateSkipped && !bulkEditFieldsToExcludeFromRevisionUpdates.has(operation.field) && rule.revision - updatedRule.revision === 0) {
      updatedRule.revision += 1;
    }
  }
  return {
    rule: updatedRule,
    ruleActions: updatedRuleActions,
    hasUpdateApiKeyOperation,
    isAttributesUpdateSkipped
  };
}
function validateScheduleInterval(context, scheduleInterval, ruleTypeId, ruleId) {
  if (!scheduleInterval) {
    return;
  }
  const isIntervalInvalid = (0, _common.parseDuration)(scheduleInterval) < context.minimumScheduleIntervalInMs;
  if (isIntervalInvalid && context.minimumScheduleInterval.enforce) {
    throw Error(`Error updating rule: the interval is less than the allowed minimum interval of ${context.minimumScheduleInterval.value}`);
  } else if (isIntervalInvalid && !context.minimumScheduleInterval.enforce) {
    context.logger.warn(`Rule schedule interval (${scheduleInterval}) for "${ruleTypeId}" rule type with ID "${ruleId}" is less than the minimum value (${context.minimumScheduleInterval.value}). Running rules at this interval may impact alerting performance. Set "xpack.alerting.rules.minimumScheduleInterval.enforce" to true to prevent such changes.`);
  }
}

/**
 * Validate that updated schedule interval is not longer than any of the existing action frequencies
 * @param schedule Schedule interval that user tries to set
 * @param actions Rule actions
 */
function validateScheduleOperation(schedule, actions, ruleId) {
  const scheduleInterval = (0, _common.parseDuration)(schedule.interval);
  const actionsWithInvalidThrottles = [];
  for (const action of actions) {
    var _action$frequency;
    // check for actions throttled shorter than the rule schedule
    if (((_action$frequency = action.frequency) === null || _action$frequency === void 0 ? void 0 : _action$frequency.notifyWhen) === _constants.ruleNotifyWhen.THROTTLE && (0, _common.parseDuration)(action.frequency.throttle) < scheduleInterval) {
      actionsWithInvalidThrottles.push(action);
    }
  }
  if (actionsWithInvalidThrottles.length > 0) {
    throw Error(`Error updating rule with ID "${ruleId}": the interval ${schedule.interval} is longer than the action frequencies`);
  }
}
async function attemptToMigrateLegacyFrequency(context, operationField, actions, rule, ruleType) {
  if (operationField !== 'actions') throw new Error('Can only perform frequency migration on an action operation');
  // Try to remove the rule-level frequency params, and then validate actions
  if (typeof rule.notifyWhen !== 'undefined') rule.notifyWhen = undefined;
  if (rule.throttle) rule.throttle = undefined;
  await (0, _lib.validateActions)(context, ruleType, {
    ...rule,
    actions
  });
  return rule;
}