"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.buildSearchBody = buildSearchBody;
exports.useEsDocSearch = useEsDocSearch;
var _react = require("react");
var _rxjs = require("rxjs");
var _ebtTools = require("@kbn/ebt-tools");
var _discoverUtils = require("@kbn/discover-utils");
var _unifiedDocViewer = require("@kbn/unified-doc-viewer");
var _plugin = require("../plugin");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * Custom react hook for querying a single doc in ElasticSearch
 */
function useEsDocSearch({
  id,
  index,
  dataView,
  esqlHit,
  onBeforeFetch,
  onProcessRecord
}) {
  const [status, setStatus] = (0, _react.useState)(_unifiedDocViewer.ElasticRequestState.Loading);
  const [hit, setHit] = (0, _react.useState)(null);
  const {
    data,
    analytics
  } = (0, _plugin.getUnifiedDocViewerServices)();
  const requestData = (0, _react.useCallback)(async () => {
    if (!index) {
      return;
    }
    const singleDocFetchingStartTime = window.performance.now();
    try {
      var _hits$hits;
      if (onBeforeFetch) {
        await onBeforeFetch();
      }
      const result = await (0, _rxjs.lastValueFrom)(data.search.search({
        params: {
          index: dataView.getIndexPattern(),
          ...buildSearchBody(id, index, dataView)
        }
      }));
      const rawResponse = result.rawResponse;
      const hits = rawResponse.hits;
      if (hits !== null && hits !== void 0 && (_hits$hits = hits.hits) !== null && _hits$hits !== void 0 && _hits$hits[0]) {
        var _hits$hits2;
        setStatus(_unifiedDocViewer.ElasticRequestState.Found);
        const record = (0, _discoverUtils.buildDataTableRecord)(hits === null || hits === void 0 ? void 0 : (_hits$hits2 = hits.hits) === null || _hits$hits2 === void 0 ? void 0 : _hits$hits2[0], dataView);
        setHit(onProcessRecord ? onProcessRecord(record) : record);
      } else {
        setStatus(_unifiedDocViewer.ElasticRequestState.NotFound);
      }
    } catch (err) {
      if (err.savedObjectId) {
        setStatus(_unifiedDocViewer.ElasticRequestState.NotFoundDataView);
      } else if (err.status === 404) {
        setStatus(_unifiedDocViewer.ElasticRequestState.NotFound);
      } else {
        setStatus(_unifiedDocViewer.ElasticRequestState.Error);
      }
    }
    if (analytics) {
      const singleDocFetchingDuration = window.performance.now() - singleDocFetchingStartTime;
      (0, _ebtTools.reportPerformanceMetricEvent)(analytics, {
        eventName: 'discoverSingleDocFetch',
        duration: singleDocFetchingDuration
      });
    }
  }, [analytics, data.search, dataView, id, index, onBeforeFetch, onProcessRecord]);
  (0, _react.useEffect)(() => {
    if (esqlHit) {
      setStatus(_unifiedDocViewer.ElasticRequestState.Found);
      setHit(esqlHit);
    } else {
      requestData();
    }
  }, [id, requestData, esqlHit]);
  return [status, hit, requestData];
}

/**
 * helper function to build a query body for Elasticsearch
 * https://www.elastic.co/guide/en/elasticsearch/reference/current//query-dsl-ids-query.html
 */
function buildSearchBody(id, index, dataView) {
  const computedFields = dataView.getComputedFields();
  const runtimeFields = computedFields.runtimeFields;
  const request = {
    query: {
      bool: {
        filter: [{
          ids: {
            values: [id]
          }
        }, {
          term: {
            _index: index
          }
        }]
      }
    },
    stored_fields: ['*'],
    script_fields: computedFields.scriptFields,
    version: true,
    _source: true,
    runtime_mappings: runtimeFields ? runtimeFields : {},
    fields: [{
      field: '*',
      include_unmapped: true
    }, ...(computedFields.docvalueFields || [])]
  };
  return request;
}