"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.DrilldownManagerState = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _useObservable = _interopRequireDefault(require("react-use/lib/useObservable"));
var _rxjs = require("rxjs");
var _react = require("react");
var _drilldown_state = require("./drilldown_state");
var _i18n = require("./i18n");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const helloMessageStorageKey = `drilldowns:hidWelcomeMessage`;
/**
 * An instance of this class holds all the state necessary for Drilldown
 * Manager. It also holds all the necessary controllers to change the state.
 *
 * `<DrilldownManager>` and other container components access this state using
 * the `useDrilldownManager()` React hook:
 *
 * ```ts
 * const state = useDrilldownManager();
 * ```
 */
class DrilldownManagerState {
  constructor(deps) {
    /**
     * Title displayed at the top of <DrilldownManager> flyout.
     */
    (0, _defineProperty2.default)(this, "title$", new _rxjs.BehaviorSubject(_i18n.txtDefaultTitle));
    /**
     * Footer displayed at the bottom of <DrilldownManager> flyout.
     */
    (0, _defineProperty2.default)(this, "footer$", new _rxjs.BehaviorSubject(null));
    /**
     * Route inside Drilldown Manager flyout that is displayed to the user. Some
     * available routes are:
     *
     * - `['create']`
     * - `['new']`
     * - `['new', 'DASHBOARD_TO_DASHBOARD_DRILLDOWN']`
     * - `['manage']`
     * - `['manage', 'yyyyyyyy-yyyy-yyyy-yyyy-yyyyyyyyyyyy']`
     */
    (0, _defineProperty2.default)(this, "route$", void 0);
    /**
     * Whether a drilldowns welcome message should be displayed to the user at
     * the very top of the drilldowns manager flyout.
     */
    (0, _defineProperty2.default)(this, "hideWelcomeMessage$", void 0);
    /**
     * Currently selected action factory (drilldown type).
     */
    (0, _defineProperty2.default)(this, "actionFactory$", void 0);
    (0, _defineProperty2.default)(this, "mapEventToDrilldownItem", event => {
      var _actionFactory$getDis;
      const actionFactory = this.deps.actionFactories.find(factory => factory.id === event.action.factoryId);
      const drilldownFactoryContext = {
        ...this.deps.placeContext,
        triggers: event.triggers
      };
      const firstTrigger = event.triggers[0];
      return {
        id: event.eventId,
        drilldownName: event.action.name,
        actionName: (_actionFactory$getDis = actionFactory === null || actionFactory === void 0 ? void 0 : actionFactory.getDisplayName(drilldownFactoryContext)) !== null && _actionFactory$getDis !== void 0 ? _actionFactory$getDis : event.action.factoryId,
        icon: actionFactory === null || actionFactory === void 0 ? void 0 : actionFactory.getIconType(drilldownFactoryContext),
        error: !actionFactory ? (0, _i18n.invalidDrilldownType)(event.action.factoryId) // this shouldn't happen for the end user, but useful during development
        : !actionFactory.isCompatibleLicense() ? _i18n.insufficientLicenseLevel : undefined,
        triggers: event.triggers.map(trigger => this.deps.getTrigger(trigger)),
        triggerIncompatible: !this.deps.triggers.find(t => t === firstTrigger)
      };
    });
    (0, _defineProperty2.default)(this, "events$", void 0);
    /**
     * State for each drilldown type used for new drilldown creation, so when user
     * switched between drilldown types the configuration of the previous
     * drilldown is preserved.
     */
    (0, _defineProperty2.default)(this, "drilldownStateByFactoryId", new Map());
    /**
     * Whether user can unlock more drilldown types if they subscribe to a higher
     * license tier.
     */
    (0, _defineProperty2.default)(this, "canUnlockMoreDrilldowns", void 0);
    /**
     * Used to show cloning success notification.
     */
    (0, _defineProperty2.default)(this, "lastCloneRecord", null);
    /**
     * Callback called to hide drilldowns welcome message, and remember in local
     * storage that user opted to hide this message.
     */
    (0, _defineProperty2.default)(this, "hideWelcomeMessage", () => {
      this.hideWelcomeMessage$.next(true);
      this.deps.storage.set(helloMessageStorageKey, true);
    });
    /**
     * Close the drilldown flyout.
     */
    (0, _defineProperty2.default)(this, "close", () => {
      this.deps.onClose();
    });
    /**
     * Deletes a list of drilldowns and shows toast notifications to the user.
     *
     * @param ids Drilldown IDs.
     */
    (0, _defineProperty2.default)(this, "onDelete", ids => {
      (async () => {
        const {
          dynamicActionManager,
          toastService
        } = this.deps;
        try {
          await dynamicActionManager.deleteEvents(ids);
          this.deps.toastService.addSuccess(ids.length === 1 ? {
            title: _i18n.toastDrilldownDeleted.title,
            text: _i18n.toastDrilldownDeleted.text
          } : {
            title: _i18n.toastDrilldownsDeleted.title(ids.length),
            text: _i18n.toastDrilldownsDeleted.text
          });
        } catch (error) {
          toastService.addError(error, {
            title: _i18n.toastDrilldownsCRUDError
          });
        }
      })().catch(console.error); // eslint-disable-line no-console
    });
    /**
     * Clone a list of selected templates.
     */
    (0, _defineProperty2.default)(this, "onClone", async templateIds => {
      const {
        templates
      } = this.deps;
      if (!templates) return;
      const templatesToClone = templateIds.map(templateId => templates.find(({
        id
      }) => id === templateId)).filter(Boolean);
      for (const template of templatesToClone) {
        await this.cloneTemplate(template);
      }
      this.lastCloneRecord = {
        time: Date.now(),
        templateIds
      };
      this.setRoute(['manage']);
    });
    (0, _defineProperty2.default)(this, "onCreateFromTemplate", async templateId => {
      const {
        templates
      } = this.deps;
      if (!templates) return;
      const template = templates.find(({
        id
      }) => id === templateId);
      if (!template) return;
      const actionFactory = this.deps.actionFactories.find(({
        id
      }) => id === template.factoryId);
      if (!actionFactory) return;
      this.setActionFactory(actionFactory);
      const drilldownState = this.getDrilldownState();
      if (drilldownState) {
        drilldownState.setName(this.pickName(template.name));
        drilldownState.setTriggers(template.triggers);
        drilldownState.setConfig(template.config);
      }
    });
    (0, _defineProperty2.default)(this, "onCreateFromDrilldown", async eventId => {
      const {
        dynamicActionManager
      } = this.deps;
      const {
        events
      } = dynamicActionManager.state.get();
      const event = events.find(ev => ev.eventId === eventId);
      if (!event) return;
      const actionFactory = this.deps.actionFactories.find(({
        id
      }) => id === event.action.factoryId);
      if (!actionFactory) return;
      this.setActionFactory(actionFactory);
      const drilldownState = this.getDrilldownState();
      if (drilldownState) {
        drilldownState.setName(this.pickName(event.action.name));
        drilldownState.setTriggers(event.triggers);
        drilldownState.setConfig(event.action.config);
      }
    });
    // Below are convenience React hooks for consuming observables in connected
    // React components.
    (0, _defineProperty2.default)(this, "useTitle", () => (0, _useObservable.default)(this.title$, this.title$.getValue()));
    (0, _defineProperty2.default)(this, "useFooter", () => (0, _useObservable.default)(this.footer$, this.footer$.getValue()));
    (0, _defineProperty2.default)(this, "useRoute", () => (0, _useObservable.default)(this.route$, this.route$.getValue()));
    (0, _defineProperty2.default)(this, "useWelcomeMessage", () => (0, _useObservable.default)(this.hideWelcomeMessage$, this.hideWelcomeMessage$.getValue()));
    (0, _defineProperty2.default)(this, "useActionFactory", () => (0, _useObservable.default)(this.actionFactory$, this.actionFactory$.getValue()));
    (0, _defineProperty2.default)(this, "useEvents", () => (0, _useObservable.default)(this.events$, this.events$.getValue()));
    (0, _defineProperty2.default)(this, "useCompatibleActionFactories", context => (0, _useObservable.default)((0, _react.useMemo)(() => this.getCompatibleActionFactories(context), [context]), undefined));
    this.deps = deps;
    const hideWelcomeMessage = deps.storage.get(helloMessageStorageKey);
    this.hideWelcomeMessage$ = new _rxjs.BehaviorSubject(hideWelcomeMessage !== null && hideWelcomeMessage !== void 0 ? hideWelcomeMessage : false);
    this.canUnlockMoreDrilldowns = deps.actionFactories.some(factory => !factory.isCompatibleLicense);
    this.events$ = new _rxjs.BehaviorSubject(this.deps.dynamicActionManager.state.get().events.map(this.mapEventToDrilldownItem));
    deps.dynamicActionManager.state.state$.pipe((0, _rxjs.map)(state => state.events.map(this.mapEventToDrilldownItem))).subscribe(this.events$);
    let {
      initialRoute = ''
    } = deps;
    if (!initialRoute) initialRoute = 'manage';else if (initialRoute[0] === '/') initialRoute = initialRoute.substr(1);
    this.route$ = new _rxjs.BehaviorSubject(initialRoute.split('/'));
    this.actionFactory$ = new _rxjs.BehaviorSubject(this.getActiveActionFactory());
    this.route$.pipe((0, _rxjs.map)(() => this.getActiveActionFactory())).subscribe(this.actionFactory$);
  }

  /**
   * Set flyout main heading text.
   * @param title New title.
   */
  setTitle(title) {
    this.title$.next(title);
  }

  /**
   * Set the new flyout footer that renders at the very bottom of the Drilldown
   * Manager flyout.
   * @param footer New title.
   */
  setFooter(footer) {
    this.footer$.next(footer);
  }

  /**
   * Set the flyout main heading back to its default state.
   */
  resetTitle() {
    this.setTitle(_i18n.txtDefaultTitle);
  }

  /**
   * Change the screen of Drilldown Manager.
   */
  setRoute(route) {
    if (route[0] === 'manage') this.deps.closeAfterCreate = false;
    this.route$.next(route);
  }
  /**
   * Select a different action factory.
   */
  setActionFactory(actionFactory) {
    if (!actionFactory) {
      const route = this.route$.getValue();
      if (route[0] === 'new' && route.length > 1) this.setRoute(['new']);
      return;
    }
    if (!this.drilldownStateByFactoryId.has(actionFactory.id)) {
      const oldActionFactory = this.getActiveActionFactory();
      const oldDrilldownState = !!oldActionFactory ? this.drilldownStateByFactoryId.get(oldActionFactory.id) : undefined;
      const context = this.getActionFactoryContext();
      const drilldownState = new _drilldown_state.DrilldownState({
        factory: actionFactory,
        placeTriggers: this.deps.triggers,
        placeContext: this.deps.placeContext || {},
        name: this.pickName(!!oldDrilldownState ? oldDrilldownState.name$.getValue() : actionFactory.getDisplayName(this.getActionFactoryContext())),
        triggers: [],
        config: actionFactory.createConfig(context)
      });
      this.drilldownStateByFactoryId.set(actionFactory.id, drilldownState);
    }
    this.route$.next(['new', actionFactory.id]);
  }
  getActiveActionFactory() {
    const [step1, id] = this.route$.getValue();
    if (step1 !== 'new' || !id) return undefined;
    return this.deps.actionFactories.find(factory => factory.id === id);
  }
  /**
   * Get action factory context, which also contains a custom place context
   * provided by the user who triggered rendering of the <DrilldownManager>.
   */
  getActionFactoryContext() {
    var _this$deps$placeConte;
    const placeContext = (_this$deps$placeConte = this.deps.placeContext) !== null && _this$deps$placeConte !== void 0 ? _this$deps$placeConte : [];
    const context = {
      ...placeContext,
      triggers: []
    };
    return context;
  }
  getCompatibleActionFactories(context) {
    const compatibleActionFactories$ = new _rxjs.BehaviorSubject(undefined);
    Promise.allSettled(this.deps.actionFactories.map(factory => factory.isCompatible(context))).then(factoryCompatibility => {
      compatibleActionFactories$.next(this.deps.actionFactories.filter((_factory, i) => {
        const result = factoryCompatibility[i];
        // treat failed isCompatible checks as non-compatible
        return result.status === 'fulfilled' && result.value;
      }));
    });
    return compatibleActionFactories$.asObservable();
  }

  /**
   * Get state object of the drilldown which is currently being created.
   */
  getDrilldownState() {
    const actionFactory = this.getActiveActionFactory();
    if (!actionFactory) return undefined;
    const drilldownState = this.drilldownStateByFactoryId.get(actionFactory.id);
    return drilldownState;
  }

  /**
   * Called when user presses "Create drilldown" button to save the
   * currently edited drilldown.
   */
  async createDrilldown() {
    const {
      dynamicActionManager,
      toastService
    } = this.deps;
    const drilldownState = this.getDrilldownState();
    if (!drilldownState) return;
    try {
      const event = drilldownState.serialize();
      const triggers = drilldownState.triggers$.getValue();
      await dynamicActionManager.createEvent(event, triggers);
      toastService.addSuccess({
        title: _i18n.toastDrilldownCreated.title(drilldownState.name$.getValue()),
        text: _i18n.toastDrilldownCreated.text
      });
      this.drilldownStateByFactoryId.delete(drilldownState.factory.id);
      if (this.deps.closeAfterCreate) {
        this.deps.onClose();
      } else {
        this.setRoute(['manage']);
      }
    } catch (error) {
      toastService.addError(error, {
        title: _i18n.toastDrilldownsCRUDError
      });
      throw error;
    }
  }
  async cloneTemplate(template) {
    const {
      dynamicActionManager
    } = this.deps;
    const name = this.pickName(template.name);
    const action = {
      factoryId: template.factoryId,
      name,
      config: template.config || {}
    };
    await dynamicActionManager.createEvent(action, template.triggers);
  }

  /**
   * Checks if drilldown with such a name already exists.
   */
  hasDrilldownWithName(name) {
    const {
      events
    } = this.deps.dynamicActionManager.state.get();
    for (const event of events) if (event.action.name === name) return true;
    return false;
  }

  /**
   * Picks a unique name for the cloned drilldown. Adds "(copy)", "(copy 1)",
   * "(copy 2)", etc. if drilldown with such name already exists.
   */
  pickName(name) {
    if (this.hasDrilldownWithName(name)) {
      const matches = name.match(/(.*) (\(copy[^\)]*\))/);
      if (matches) name = matches[1];
      for (let i = 0; i < 100; i++) {
        const proposedName = !i ? `${name} (copy)` : `${name} (copy ${i})`;
        const exists = this.hasDrilldownWithName(proposedName);
        if (!exists) return proposedName;
      }
    }
    return name;
  }
  /**
   * Returns the state object of an existing drilldown for editing purposes.
   *
   * @param eventId ID of the saved dynamic action event.
   */
  createEventDrilldownState(eventId) {
    const {
      dynamicActionManager,
      actionFactories,
      triggers: placeTriggers
    } = this.deps;
    const {
      events
    } = dynamicActionManager.state.get();
    const event = events.find(ev => ev.eventId === eventId);
    if (!event) return null;
    const factory = actionFactories.find(({
      id
    }) => id === event.action.factoryId);
    if (!factory) return null;
    const {
      action,
      triggers
    } = event;
    const {
      name,
      config
    } = action;
    const state = new _drilldown_state.DrilldownState({
      factory,
      placeContext: this.getActionFactoryContext(),
      placeTriggers,
      name,
      config,
      triggers
    });
    return state;
  }

  /**
   * Save edits to an existing drilldown.
   *
   * @param eventId ID of the saved dynamic action event.
   * @param drilldownState Latest state of the drilldown as edited by the user.
   */
  async updateEvent(eventId, drilldownState) {
    const {
      dynamicActionManager,
      toastService
    } = this.deps;
    const action = drilldownState.serialize();
    try {
      await dynamicActionManager.updateEvent(eventId, action, drilldownState.triggers$.getValue());
      toastService.addSuccess({
        title: _i18n.toastDrilldownEdited.title(action.name),
        text: _i18n.toastDrilldownEdited.text
      });
      this.setRoute(['manage']);
    } catch (error) {
      toastService.addError(error, {
        title: _i18n.toastDrilldownsCRUDError
      });
      throw error;
    }
  }
}
exports.DrilldownManagerState = DrilldownManagerState;