"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.TaskTypeDictionary = exports.SHARED_CONCURRENCY_TASKS = exports.REMOVED_TYPES = void 0;
exports.sanitizeTaskDefinitions = sanitizeTaskDefinitions;
exports.sharedConcurrencyTaskTypes = sharedConcurrencyTaskTypes;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _task = require("./task");
var _constants = require("./constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Types that are no longer registered and will be marked as unregistered
 */
const REMOVED_TYPES = exports.REMOVED_TYPES = [
// for testing
'sampleTaskRemovedType',
// deprecated in https://github.com/elastic/kibana/pull/121442
'alerting:siem.signals', 'search_sessions_monitor', 'search_sessions_cleanup', 'search_sessions_expire', 'cleanup_failed_action_executions', 'reports:monitor',
// deprecated in https://github.com/elastic/kibana/pull/216916
'obs-ai-assistant:knowledge-base-migration'];
const SHARED_CONCURRENCY_TASKS = exports.SHARED_CONCURRENCY_TASKS = [
// for testing
['sampleTaskSharedConcurrencyType1', 'sampleTaskSharedConcurrencyType2'],
// reporting
['report:execute', 'report:execute-scheduled']];

/**
 * Defines a task which can be scheduled and run by the Kibana
 * task manager.
 */

/**
 * A mapping of task type id to the task definition.
 */

class TaskTypeDictionary {
  constructor(logger) {
    (0, _defineProperty2.default)(this, "definitions", new Map());
    (0, _defineProperty2.default)(this, "logger", void 0);
    this.logger = logger;
  }
  [Symbol.iterator]() {
    return this.definitions.entries();
  }
  getAllTypes() {
    return [...this.definitions.keys()];
  }
  getAllDefinitions() {
    return [...this.definitions.values()];
  }
  has(type) {
    return this.definitions.has(type);
  }
  size() {
    return this.definitions.size;
  }
  get(type) {
    return this.definitions.get(type);
  }
  ensureHas(type) {
    if (!this.has(type)) {
      throw new Error(`Unsupported task type "${type}". Supported types are ${this.getAllTypes().join(', ')}`);
    }
  }

  /**
   * Method for allowing consumers to register task definitions into the system.
   * @param taskDefinitions - The Kibana task definitions dictionary
   */
  registerTaskDefinitions(taskDefinitions) {
    const taskTypesToRegister = Object.keys(taskDefinitions);
    const duplicate = taskTypesToRegister.find(type => this.definitions.has(type));
    if (duplicate) {
      throw new Error(`Task ${duplicate} is already defined!`);
    }
    const invalidTaskType = taskTypesToRegister.find(type => type.includes(','));
    if (invalidTaskType) {
      throw new Error(`Task type "${invalidTaskType}" cannot contain a comma.`);
    }
    const removed = taskTypesToRegister.find(type => REMOVED_TYPES.indexOf(type) >= 0);
    if (removed) {
      throw new Error(`Task ${removed} has been removed from registration!`);
    }
    for (const taskType of taskTypesToRegister) {
      if (taskDefinitions[taskType].maxConcurrency !== undefined) {
        if (!_constants.CONCURRENCY_ALLOW_LIST_BY_TASK_TYPE.includes(taskType)) {
          // maxConcurrency is designed to limit how many tasks of the same type a single Kibana
          // instance should run at a time. Meaning if you have 8 Kibanas running, you will still
          // see up to 8 tasks running at a time but one per Kibana instance. This is helpful for
          // reporting purposes but not for many other cases and are better off not setting this value.
          throw new Error(`maxConcurrency setting isn't allowed for task type: ${taskType}`);
        }

        // if this task type shares concurrency with another task type and both have been
        // registered, throw an error if their maxConcurrency values are different
        this.verifySharedConcurrencyAndCost(taskType, taskDefinitions[taskType].maxConcurrency, taskDefinitions[taskType].cost);
      }
    }
    try {
      for (const definition of sanitizeTaskDefinitions(taskDefinitions)) {
        this.definitions.set(definition.type, definition);
      }
    } catch (e) {
      this.logger.error(`Could not sanitize task definitions: ${e.message}`);
    }
  }
  verifySharedConcurrencyAndCost(taskType, maxConcurrency, cost) {
    const shared = sharedConcurrencyTaskTypes(taskType);
    if (shared) {
      const otherTaskTypes = shared.filter(type => type !== taskType);
      for (const otherTaskType of otherTaskTypes) {
        const otherTaskDef = this.definitions.get(otherTaskType);
        if (otherTaskDef && otherTaskDef.maxConcurrency !== maxConcurrency) {
          throw new Error(`Task type "${taskType}" shares concurrency limits with ${otherTaskType} but has a different maxConcurrency.`);
        }
        if (otherTaskDef && otherTaskDef.cost !== cost) {
          throw new Error(`Task type "${taskType}" shares concurrency limits with ${otherTaskType} but has a different cost.`);
        }
      }
    }
  }
}

/**
 * Sanitizes the system's task definitions. Task definitions have optional properties, and
 * this ensures they all are given a reasonable default.
 *
 * @param taskDefinitions - The Kibana task definitions dictionary
 */
exports.TaskTypeDictionary = TaskTypeDictionary;
function sanitizeTaskDefinitions(taskDefinitions) {
  return Object.entries(taskDefinitions).map(([type, rawDefinition]) => {
    return _task.taskDefinitionSchema.validate({
      type,
      ...rawDefinition
    });
  });
}
function sharedConcurrencyTaskTypes(taskType) {
  return SHARED_CONCURRENCY_TASKS.find(tasks => tasks.includes(taskType));
}