"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.TASK_ID = exports.SCHEDULE_INTERVAL = void 0;
exports.registerMarkRemovedTasksAsUnrecognizedDefinition = registerMarkRemovedTasksAsUnrecognizedDefinition;
exports.scheduleMarkRemovedTasksAsUnrecognizedDefinition = scheduleMarkRemovedTasksAsUnrecognizedDefinition;
exports.taskRunner = taskRunner;
var _task = require("../task");
var _task_type_dictionary = require("../task_type_dictionary");
var _constants = require("../constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const TASK_ID = exports.TASK_ID = 'mark_removed_tasks_as_unrecognized';
const TASK_TYPE = `task_manager:${TASK_ID}`;
const SCHEDULE_INTERVAL = exports.SCHEDULE_INTERVAL = '1h';
async function scheduleMarkRemovedTasksAsUnrecognizedDefinition(logger, taskScheduling) {
  try {
    await taskScheduling.ensureScheduled({
      id: TASK_ID,
      taskType: TASK_TYPE,
      schedule: {
        interval: SCHEDULE_INTERVAL
      },
      state: {},
      params: {}
    });
  } catch (e) {
    logger.error(`Error scheduling ${TASK_ID} task, received ${e.message}`);
  }
}
function registerMarkRemovedTasksAsUnrecognizedDefinition(logger, coreStartServices, taskTypeDictionary) {
  taskTypeDictionary.registerTaskDefinitions({
    [TASK_TYPE]: {
      title: 'Mark removed tasks as unrecognized',
      createTaskRunner: taskRunner(logger, coreStartServices)
    }
  });
}
function taskRunner(logger, coreStartServices) {
  return () => {
    return {
      async run() {
        try {
          const [{
            elasticsearch
          }] = await coreStartServices();
          const esClient = elasticsearch.client.asInternalUser;
          const removedTasks = await queryForRemovedTasks(esClient);
          if (removedTasks.length > 0) {
            await updateTasksToBeUnrecognized(esClient, logger, removedTasks);
          }
          return {
            state: {},
            schedule: {
              interval: SCHEDULE_INTERVAL
            }
          };
        } catch (e) {
          logger.error(`Failed to mark removed tasks as unrecognized. Error: ${e.message}`);
          return {
            state: {},
            schedule: {
              interval: SCHEDULE_INTERVAL
            }
          };
        }
      }
    };
  };
}
async function queryForRemovedTasks(esClient) {
  const result = await esClient.search({
    index: _constants.TASK_MANAGER_INDEX,
    size: 100,
    _source: false,
    query: {
      bool: {
        must: [{
          terms: {
            'task.taskType': _task_type_dictionary.REMOVED_TYPES
          }
        }]
      }
    }
  });
  return result.hits.hits;
}
async function updateTasksToBeUnrecognized(esClient, logger, removedTasks) {
  const bulkBody = [];
  for (const task of removedTasks) {
    bulkBody.push({
      update: {
        _id: task._id
      }
    });
    bulkBody.push({
      doc: {
        task: {
          status: _task.TaskStatus.Unrecognized
        }
      }
    });
  }
  let removedCount = 0;
  try {
    const removeResults = await esClient.bulk({
      index: _constants.TASK_MANAGER_INDEX,
      refresh: false,
      body: bulkBody
    });
    for (const removeResult of removeResults.items) {
      var _removeResult$update;
      if (!removeResult.update || !removeResult.update._id) {
        logger.warn(`Error updating task with unknown to mark as unrecognized - malformed response`);
      } else if ((_removeResult$update = removeResult.update) !== null && _removeResult$update !== void 0 && _removeResult$update.error) {
        logger.warn(`Error updating task ${removeResult.update._id} to mark as unrecognized - ${JSON.stringify(removeResult.update.error)}`);
      } else {
        removedCount++;
      }
    }
    logger.debug(`Marked ${removedCount} removed tasks as unrecognized`);
  } catch (err) {
    // don't worry too much about errors, we'll try again next time
    logger.warn(`Error updating tasks to mark as unrecognized: ${err}`);
  }
}