"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.runSynPrivateLocationMonitorsTaskSoon = exports.resetSyncPrivateCleanUpState = exports.SyncPrivateLocationMonitorsTask = exports.PRIVATE_LOCATIONS_SYNC_TASK_ID = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _constants = require("@kbn/spaces-plugin/common/constants");
var _moment = _interopRequireDefault(require("moment"));
var _common = require("@kbn/alerting-plugin/common");
var _pRetry = _interopRequireDefault(require("p-retry"));
var _clean_up_task = require("../synthetics_service/private_location/clean_up_task");
var _saved_objects = require("../../common/types/saved_objects");
var _utils = require("../synthetics_service/utils");
var _monitor_config_repository = require("../services/monitor_config_repository");
var _get_private_locations = require("../synthetics_service/get_private_locations");
var _format_configs = require("../synthetics_service/formatters/public_formatters/format_configs");
var _synthetics_private_location = require("../synthetics_service/private_location/synthetics_private_location");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const TASK_TYPE = 'Synthetics:Sync-Private-Location-Monitors';
const PRIVATE_LOCATIONS_SYNC_TASK_ID = exports.PRIVATE_LOCATIONS_SYNC_TASK_ID = `${TASK_TYPE}-single-instance`;
const TASK_SCHEDULE = '10m';
class SyncPrivateLocationMonitorsTask {
  constructor(serverSetup, _taskManager, syntheticsMonitorClient) {
    (0, _defineProperty2.default)(this, "start", async () => {
      const {
        pluginsStart: {
          taskManager
        }
      } = this.serverSetup;
      this.debugLog(`Scheduling private location task`);
      await taskManager.ensureScheduled({
        id: PRIVATE_LOCATIONS_SYNC_TASK_ID,
        state: {},
        schedule: {
          interval: TASK_SCHEDULE
        },
        taskType: TASK_TYPE,
        params: {}
      });
      this.debugLog(`Sync private location monitors task scheduled successfully`);
    });
    (0, _defineProperty2.default)(this, "hasAnyDataChanged", async ({
      taskState,
      soClient
    }) => {
      const {
        lastTotalParams,
        lastTotalMWs,
        lastStartedAt
      } = taskState;
      const {
        totalParams,
        hasParamsChanges
      } = await this.hasAnyParamChanged({
        soClient,
        lastStartedAt,
        lastTotalParams
      });
      const {
        totalMWs,
        hasMWsChanged
      } = await this.hasMWsChanged({
        soClient,
        lastStartedAt,
        lastTotalMWs
      });
      taskState.lastTotalParams = totalParams;
      taskState.lastTotalMWs = totalMWs;
      const hasDataChanged = hasMWsChanged || hasParamsChanges;
      return {
        hasDataChanged
      };
    });
    (0, _defineProperty2.default)(this, "debugLog", message => {
      this.serverSetup.logger.debug(`[SyncPrivateLocationMonitorsTask] ${message}`);
    });
    this.serverSetup = serverSetup;
    this.taskManager = _taskManager;
    this.syntheticsMonitorClient = syntheticsMonitorClient;
    _taskManager.registerTaskDefinitions({
      [TASK_TYPE]: {
        title: 'Synthetics Sync Global Params Task',
        description: 'This task is executed so that we can sync private location monitors for example when global params are updated',
        timeout: '5m',
        maxAttempts: 1,
        createTaskRunner: ({
          taskInstance
        }) => {
          return {
            run: async () => {
              return this.runTask({
                taskInstance
              });
            }
          };
        }
      }
    });
  }
  async runTask({
    taskInstance
  }) {
    const {
      coreStart: {
        savedObjects
      },
      encryptedSavedObjects,
      logger
    } = this.serverSetup;
    const lastStartedAt = taskInstance.state.lastStartedAt || (0, _moment.default)().subtract(10, 'minute').toISOString();
    const startedAt = taskInstance.startedAt || new Date();
    const taskState = {
      lastStartedAt,
      startedAt: startedAt.toISOString(),
      lastTotalParams: taskInstance.state.lastTotalParams || 0,
      lastTotalMWs: taskInstance.state.lastTotalMWs || 0,
      hasAlreadyDoneCleanup: taskInstance.state.hasAlreadyDoneCleanup || false,
      maxCleanUpRetries: taskInstance.state.maxCleanUpRetries || 3
    };
    try {
      this.debugLog(`Syncing private location monitors, current task state is ${JSON.stringify(taskState)}`);
      const soClient = savedObjects.createInternalRepository([_common.MAINTENANCE_WINDOW_SAVED_OBJECT_TYPE]);
      const {
        performSync
      } = await this.cleanUpDuplicatedPackagePolicies(soClient, taskState);
      const allPrivateLocations = await (0, _get_private_locations.getPrivateLocations)(soClient, _constants.ALL_SPACES_ID);
      const {
        hasDataChanged
      } = await this.hasAnyDataChanged({
        soClient,
        taskState
      });
      if (hasDataChanged || performSync) {
        if (hasDataChanged) {
          this.debugLog(`Syncing private location monitors because data has changed`);
        } else {
          this.debugLog(`Syncing private location monitors because cleanup performed a change`);
        }
        if (allPrivateLocations.length > 0) {
          await this.syncGlobalParams({
            allPrivateLocations,
            soClient,
            encryptedSavedObjects
          });
        }
        this.debugLog(`Sync of private location monitors succeeded`);
      } else {
        this.debugLog(`No data has changed since last run ${lastStartedAt}, skipping sync of private location monitors`);
      }
    } catch (error) {
      logger.error(`Sync of private location monitors failed: ${error.message}`);
      return {
        error,
        state: taskState,
        schedule: {
          interval: TASK_SCHEDULE
        }
      };
    }
    return {
      state: taskState,
      schedule: {
        interval: TASK_SCHEDULE
      }
    };
  }
  async syncGlobalParams({
    allPrivateLocations,
    encryptedSavedObjects,
    soClient
  }) {
    const {
      privateLocationAPI
    } = this.syntheticsMonitorClient;
    const {
      configsBySpaces,
      paramsBySpace,
      spaceIds,
      maintenanceWindows
    } = await this.getAllMonitorConfigs({
      encryptedSavedObjects,
      soClient
    });
    return this.serverSetup.fleet.runWithCache(async () => {
      for (const spaceId of spaceIds) {
        var _monitors$length;
        const privateConfigs = [];
        const monitors = configsBySpaces[spaceId];
        this.debugLog(`Processing spaceId: ${spaceId}, monitors count: ${(_monitors$length = monitors === null || monitors === void 0 ? void 0 : monitors.length) !== null && _monitors$length !== void 0 ? _monitors$length : 0}`);
        if (!monitors) {
          continue;
        }
        for (const monitor of monitors) {
          const {
            privateLocations
          } = this.parseLocations(monitor);
          if (privateLocations.length > 0) {
            privateConfigs.push({
              config: monitor,
              globalParams: paramsBySpace[spaceId]
            });
          }
        }
        if (privateConfigs.length > 0) {
          this.debugLog(`Syncing private configs for spaceId: ${spaceId}, privateConfigs count: ${privateConfigs.length}`);
          await privateLocationAPI.editMonitors(privateConfigs, allPrivateLocations, spaceId, maintenanceWindows);
        } else {
          this.debugLog(`No privateConfigs to sync for spaceId: ${spaceId}`);
        }
      }
    });
  }
  async getAllMonitorConfigs({
    soClient,
    encryptedSavedObjects
  }) {
    const {
      syntheticsService
    } = this.syntheticsMonitorClient;
    const paramsBySpacePromise = syntheticsService.getSyntheticsParams({
      spaceId: _constants.ALL_SPACES_ID
    });
    const maintenanceWindowsPromise = syntheticsService.getMaintenanceWindows();
    const monitorConfigRepository = new _monitor_config_repository.MonitorConfigRepository(soClient, encryptedSavedObjects.getClient());
    const monitorsPromise = monitorConfigRepository.findDecryptedMonitors({
      spaceId: _constants.ALL_SPACES_ID
    });
    const [paramsBySpace, monitors, maintenanceWindows] = await Promise.all([paramsBySpacePromise, monitorsPromise, maintenanceWindowsPromise]);
    return {
      ...this.mixParamsWithMonitors(monitors, paramsBySpace),
      paramsBySpace,
      maintenanceWindows
    };
  }
  parseLocations(config) {
    const {
      locations
    } = config;
    const privateLocations = locations.filter(loc => !loc.isServiceManaged);
    const publicLocations = locations.filter(loc => loc.isServiceManaged);
    return {
      privateLocations,
      publicLocations
    };
  }
  mixParamsWithMonitors(monitors, paramsBySpace) {
    const configsBySpaces = {};
    const spaceIds = new Set();
    for (const monitor of monitors) {
      var _monitor$namespaces;
      const spaceId = (_monitor$namespaces = monitor.namespaces) === null || _monitor$namespaces === void 0 ? void 0 : _monitor$namespaces[0];
      if (!spaceId) {
        continue;
      }
      spaceIds.add(spaceId);
      const normalizedMonitor = (0, _utils.normalizeSecrets)(monitor).attributes;
      const {
        str: paramsString
      } = (0, _format_configs.mixParamsWithGlobalParams)(paramsBySpace[spaceId], normalizedMonitor);
      if (!configsBySpaces[spaceId]) {
        configsBySpaces[spaceId] = [];
      }
      configsBySpaces[spaceId].push((0, _format_configs.formatHeartbeatRequest)({
        spaceId,
        monitor: normalizedMonitor,
        configId: monitor.id
      }, paramsString));
    }
    return {
      configsBySpaces,
      spaceIds
    };
  }
  async hasAnyParamChanged({
    soClient,
    lastStartedAt,
    lastTotalParams
  }) {
    const {
      logger
    } = this.serverSetup;
    const [editedParams, totalParams] = await Promise.all([soClient.find({
      type: _saved_objects.syntheticsParamType,
      perPage: 0,
      namespaces: [_constants.ALL_SPACES_ID],
      filter: `synthetics-param.updated_at > "${lastStartedAt}"`,
      fields: []
    }), soClient.find({
      type: _saved_objects.syntheticsParamType,
      perPage: 0,
      namespaces: [_constants.ALL_SPACES_ID],
      fields: []
    })]);
    logger.debug(`Found ${editedParams.total} params updated and ${totalParams.total} total params`);
    const updatedParams = editedParams.total;
    const noOfParams = totalParams.total;
    const hasParamsChanges = updatedParams > 0 || noOfParams !== lastTotalParams;
    return {
      hasParamsChanges,
      updatedParams: editedParams.total,
      totalParams: noOfParams
    };
  }
  async hasMWsChanged({
    soClient,
    lastStartedAt,
    lastTotalMWs
  }) {
    const {
      logger
    } = this.serverSetup;
    const [editedMWs, totalMWs] = await Promise.all([soClient.find({
      type: _common.MAINTENANCE_WINDOW_SAVED_OBJECT_TYPE,
      perPage: 0,
      namespaces: [_constants.ALL_SPACES_ID],
      filter: `${_common.MAINTENANCE_WINDOW_SAVED_OBJECT_TYPE}.updated_at > "${lastStartedAt}"`,
      fields: []
    }), soClient.find({
      type: _common.MAINTENANCE_WINDOW_SAVED_OBJECT_TYPE,
      perPage: 0,
      namespaces: [_constants.ALL_SPACES_ID],
      fields: []
    })]);
    logger.debug(`Found ${editedMWs.total} maintenance windows updated and ${totalMWs.total} total maintenance windows`);
    const updatedMWs = editedMWs.total;
    const noOfMWs = totalMWs.total;
    const hasMWsChanged = updatedMWs > 0 || noOfMWs !== lastTotalMWs;
    return {
      hasMWsChanged,
      updatedMWs,
      totalMWs: noOfMWs
    };
  }
  async cleanUpDuplicatedPackagePolicies(soClient, taskState) {
    let performSync = false;
    if (taskState.hasAlreadyDoneCleanup) {
      this.debugLog('Skipping cleanup of duplicated package policies as it has already been done once');
      return {
        performSync
      };
    } else if (taskState.maxCleanUpRetries <= 0) {
      this.debugLog('Skipping cleanup of duplicated package policies as max retries have been reached');
      taskState.hasAlreadyDoneCleanup = true;
      taskState.maxCleanUpRetries = 3;
      return {
        performSync
      };
    }
    this.debugLog('Starting cleanup of duplicated package policies');
    const {
      fleet
    } = this.serverSetup.pluginsStart;
    const {
      logger
    } = this.serverSetup;
    try {
      var _this$serverSetup$cor, _this$serverSetup$cor2;
      const esClient = (_this$serverSetup$cor = this.serverSetup.coreStart) === null || _this$serverSetup$cor === void 0 ? void 0 : (_this$serverSetup$cor2 = _this$serverSetup$cor.elasticsearch) === null || _this$serverSetup$cor2 === void 0 ? void 0 : _this$serverSetup$cor2.client.asInternalUser;
      const finder = soClient.createPointInTimeFinder({
        type: _saved_objects.syntheticsMonitorSOTypes,
        fields: ['id', 'name', 'locations', 'origin'],
        namespaces: ['*']
      });
      const privateLocationAPI = new _synthetics_private_location.SyntheticsPrivateLocation(this.serverSetup);
      const expectedPackagePolicies = new Set();
      for await (const result of finder.find()) {
        result.saved_objects.forEach(monitor => {
          var _monitor$attributes$l;
          (_monitor$attributes$l = monitor.attributes.locations) === null || _monitor$attributes$l === void 0 ? void 0 : _monitor$attributes$l.forEach(location => {
            var _monitor$namespaces2;
            const spaceId = (_monitor$namespaces2 = monitor.namespaces) === null || _monitor$namespaces2 === void 0 ? void 0 : _monitor$namespaces2[0];
            if (!location.isServiceManaged && spaceId) {
              const policyId = privateLocationAPI.getPolicyId({
                origin: monitor.attributes.origin,
                id: monitor.id
              }, location.id, spaceId);
              expectedPackagePolicies.add(policyId);
            }
          });
        });
      }
      finder.close().catch(() => {});
      const packagePoliciesKuery = (0, _clean_up_task.getFilterForTestNowRun)(true);
      const policiesIterator = await fleet.packagePolicyService.fetchAllItemIds(soClient, {
        kuery: packagePoliciesKuery,
        spaceIds: ['*'],
        perPage: 100
      });
      const packagePoliciesToDelete = [];
      for await (const packagePoliciesIds of policiesIterator) {
        for (const packagePolicyId of packagePoliciesIds) {
          if (!expectedPackagePolicies.has(packagePolicyId)) {
            packagePoliciesToDelete.push(packagePolicyId);
          }
          // remove it from the set to mark it as found
          expectedPackagePolicies.delete(packagePolicyId);
        }
      }

      // if we have any to delete or any expected that were not found we need to perform a sync
      performSync = packagePoliciesToDelete.length > 0 || expectedPackagePolicies.size > 0;
      if (packagePoliciesToDelete.length > 0) {
        logger.info(` [PrivateLocationCleanUpTask] Found ${packagePoliciesToDelete.length} duplicate package policies to delete: ${packagePoliciesToDelete.join(', ')}`);
        await fleet.packagePolicyService.delete(soClient, esClient, packagePoliciesToDelete, {
          force: true,
          spaceIds: ['*']
        });
      }
      taskState.hasAlreadyDoneCleanup = true;
      taskState.maxCleanUpRetries = 3;
      return {
        performSync
      };
    } catch (e) {
      taskState.maxCleanUpRetries -= 1;
      if (taskState.maxCleanUpRetries <= 0) {
        this.debugLog('Skipping cleanup of duplicated package policies as max retries have been reached');
        taskState.hasAlreadyDoneCleanup = true;
        taskState.maxCleanUpRetries = 3;
      }
      logger.error('[SyncPrivateLocationMonitorsTask] Error cleaning up duplicated package policies', {
        error: e
      });
      return {
        performSync
      };
    }
  }
}
exports.SyncPrivateLocationMonitorsTask = SyncPrivateLocationMonitorsTask;
const runSynPrivateLocationMonitorsTaskSoon = async ({
  server,
  retries = 5
}) => {
  try {
    await (0, _pRetry.default)(async () => {
      const {
        logger,
        pluginsStart: {
          taskManager
        }
      } = server;
      logger.debug(`Scheduling Synthetics sync private location monitors task soon`);
      await taskManager.runSoon(PRIVATE_LOCATIONS_SYNC_TASK_ID);
      logger.debug(`Synthetics sync private location task scheduled successfully`);
    }, {
      retries
    });
  } catch (error) {
    server.logger.error(`Error scheduling Synthetics sync private location monitors task: ${error.message}`, {
      error
    });
  }
};
exports.runSynPrivateLocationMonitorsTaskSoon = runSynPrivateLocationMonitorsTaskSoon;
const resetSyncPrivateCleanUpState = async ({
  server
}) => {
  const {
    logger,
    pluginsStart: {
      taskManager
    }
  } = server;
  logger.debug(`Resetting Synthetics sync private location monitors cleanup state`);
  await taskManager.bulkUpdateState([PRIVATE_LOCATIONS_SYNC_TASK_ID], state => ({
    ...state,
    hasAlreadyDoneCleanup: false
  }));
  await runSynPrivateLocationMonitorsTaskSoon({
    server
  });
  logger.debug(`Synthetics sync private location monitors cleanup state reset successfully`);
};
exports.resetSyncPrivateCleanUpState = resetSyncPrivateCleanUpState;