"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.syncEditedMonitorBulk = exports.rollbackFailedUpdates = exports.rollbackCompletely = void 0;
var _runtime_types = require("../../../../common/runtime_types");
var _monitor_upgrade_sender = require("../../telemetry/monitor_upgrade_sender");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

// Simplify return promise type and type it with runtime_types

async function syncUpdatedMonitors({
  spaceId,
  privateLocations,
  routeContext,
  monitorsToUpdate
}) {
  const {
    syntheticsMonitorClient
  } = routeContext;
  return await syntheticsMonitorClient.editMonitors(monitorsToUpdate.map(({
    normalizedMonitor,
    decryptedPreviousMonitor
  }) => ({
    monitor: {
      ...normalizedMonitor,
      [_runtime_types.ConfigKey.CONFIG_ID]: decryptedPreviousMonitor.id,
      [_runtime_types.ConfigKey.MONITOR_QUERY_ID]: normalizedMonitor[_runtime_types.ConfigKey.CUSTOM_HEARTBEAT_ID] || decryptedPreviousMonitor.id
    },
    id: decryptedPreviousMonitor.id,
    decryptedPreviousMonitor
  })), privateLocations, spaceId);
}
const syncEditedMonitorBulk = async ({
  routeContext,
  spaceId,
  monitorsToUpdate,
  privateLocations
}) => {
  const {
    server,
    monitorConfigRepository
  } = routeContext;
  try {
    const data = monitorsToUpdate.map(({
      monitorWithRevision,
      decryptedPreviousMonitor
    }) => ({
      id: decryptedPreviousMonitor.id,
      attributes: {
        ...monitorWithRevision,
        [_runtime_types.ConfigKey.CONFIG_ID]: decryptedPreviousMonitor.id,
        [_runtime_types.ConfigKey.MONITOR_QUERY_ID]: monitorWithRevision[_runtime_types.ConfigKey.CUSTOM_HEARTBEAT_ID] || decryptedPreviousMonitor.id
      },
      previousMonitor: decryptedPreviousMonitor
    }));
    const [editedMonitorSavedObjects, editSyncResponse] = await Promise.all([monitorConfigRepository.bulkUpdate({
      monitors: data,
      namespace: spaceId !== routeContext.spaceId ? spaceId : undefined
    }), syncUpdatedMonitors({
      monitorsToUpdate,
      routeContext,
      spaceId,
      privateLocations
    })]);
    const {
      failedPolicyUpdates,
      publicSyncErrors
    } = editSyncResponse;
    monitorsToUpdate.forEach(({
      normalizedMonitor,
      decryptedPreviousMonitor
    }) => {
      const editedMonitorSavedObject = editedMonitorSavedObjects === null || editedMonitorSavedObjects === void 0 ? void 0 : editedMonitorSavedObjects.saved_objects.find(obj => obj.id === decryptedPreviousMonitor.id);
      (0, _monitor_upgrade_sender.sendTelemetryEvents)(server.logger, server.telemetry, (0, _monitor_upgrade_sender.formatTelemetryUpdateEvent)(editedMonitorSavedObject, decryptedPreviousMonitor.updated_at, server.stackVersion, Boolean(normalizedMonitor[_runtime_types.ConfigKey.SOURCE_INLINE]), publicSyncErrors));
    });
    const failedConfigs = await rollbackFailedUpdates({
      monitorsToUpdate,
      routeContext,
      failedPolicyUpdates
    });
    return {
      failedConfigs,
      errors: publicSyncErrors,
      editedMonitors: editedMonitorSavedObjects === null || editedMonitorSavedObjects === void 0 ? void 0 : editedMonitorSavedObjects.saved_objects
    };
  } catch (e) {
    await rollbackCompletely({
      routeContext,
      monitorsToUpdate
    });
    throw e;
  }
};
exports.syncEditedMonitorBulk = syncEditedMonitorBulk;
const rollbackCompletely = async ({
  routeContext,
  monitorsToUpdate
}) => {
  const {
    server,
    monitorConfigRepository
  } = routeContext;
  try {
    await monitorConfigRepository.bulkUpdate({
      monitors: monitorsToUpdate.map(({
        decryptedPreviousMonitor
      }) => ({
        id: decryptedPreviousMonitor.id,
        attributes: decryptedPreviousMonitor.attributes,
        previousMonitor: decryptedPreviousMonitor
      }))
    });
  } catch (error) {
    server.logger.error(`Unable to rollback Synthetics monitors edit, Error: ${error.message}`, {
      error
    });
  }
};
exports.rollbackCompletely = rollbackCompletely;
const rollbackFailedUpdates = async ({
  routeContext,
  failedPolicyUpdates,
  monitorsToUpdate
}) => {
  if (!failedPolicyUpdates || failedPolicyUpdates.length === 0) {
    return;
  }
  const {
    server,
    monitorConfigRepository
  } = routeContext;
  try {
    const failedConfigs = {};
    failedPolicyUpdates.forEach(({
      config,
      error
    }) => {
      if (config && config[_runtime_types.ConfigKey.CONFIG_ID]) {
        failedConfigs[config[_runtime_types.ConfigKey.CONFIG_ID]] = {
          config,
          error
        };
      }
    });
    const monitorsToRevert = monitorsToUpdate.filter(({
      decryptedPreviousMonitor
    }) => {
      return failedConfigs[decryptedPreviousMonitor.id];
    }).map(({
      decryptedPreviousMonitor
    }) => ({
      id: decryptedPreviousMonitor.id,
      attributes: decryptedPreviousMonitor.attributes,
      previousMonitor: decryptedPreviousMonitor
    }));
    if (monitorsToRevert.length > 0) {
      await monitorConfigRepository.bulkUpdate({
        monitors: monitorsToRevert
      });
    }
    return failedConfigs;
  } catch (error) {
    server.logger.error(`Unable to rollback Synthetics monitor failed updates, Error: ${error.message}`, {
      error
    });
  }
};
exports.rollbackFailedUpdates = rollbackFailedUpdates;