"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.licenseCheck = exports.createSyntheticsRouteWithAuth = void 0;
var _constants = require("../../common/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function getDefaultWriteAccessFlag(method) {
  // if the method is not GET, it defaults to requiring write access
  return method !== 'GET';
}
const createSyntheticsRouteWithAuth = routeCreator => {
  const restRoute = routeCreator();
  const {
    handler,
    method,
    path,
    options,
    writeAccess,
    ...rest
  } = restRoute;
  const licenseCheckHandler = async ({
    context,
    response,
    ...restProps
  }) => {
    const {
      statusCode,
      message
    } = licenseCheck((await context.licensing).license);
    if (statusCode === 200) {
      return handler({
        context,
        response,
        ...restProps
      });
    }
    switch (statusCode) {
      case 400:
        return response.badRequest({
          body: {
            message
          }
        });
      case 401:
        return response.unauthorized({
          body: {
            message
          }
        });
      case 403:
        return response.forbidden({
          body: {
            message
          }
        });
      default:
        throw new Error('Failed to validate the license');
    }
  };
  return {
    method,
    path,
    options,
    handler: licenseCheckHandler,
    ...rest,
    writeAccess: writeAccess !== null && writeAccess !== void 0 ? writeAccess : getDefaultWriteAccessFlag(method)
  };
};
exports.createSyntheticsRouteWithAuth = createSyntheticsRouteWithAuth;
const licenseCheck = license => {
  if (license === undefined) {
    return {
      message: _constants.LICENSE_MISSING_ERROR,
      statusCode: 400
    };
  }
  if (!license.hasAtLeast('basic')) {
    return {
      message: _constants.LICENSE_NOT_SUPPORTED_ERROR,
      statusCode: 401
    };
  }
  if (license.isActive === false) {
    return {
      message: _constants.LICENSE_NOT_ACTIVE_ERROR,
      statusCode: 403
    };
  }
  return {
    message: 'License is valid and active',
    statusCode: 200
  };
};
exports.licenseCheck = licenseCheck;