"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getPendingConfigs = exports.getMissingPingMonitorInfo = exports.getConfigStats = exports.calculateIsValidPing = void 0;
var _moment = _interopRequireDefault(require("moment"));
var _runtime_types = require("../../../../common/runtime_types");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getMissingPingMonitorInfo = ({
  monitors,
  configId,
  locationId
}) => {
  var _monitor$attributes$l;
  const monitor = monitors.find(
  // for project monitors, we can match by id or by monitor query id
  m => m.id === configId || m.attributes[_runtime_types.ConfigKey.MONITOR_QUERY_ID] === configId);
  if (!monitor) {
    // This should never happen
    return;
  }

  // For some reason 'urls' is not considered a valid attribute in the monitor attributes, there's probably a problem with the EncryptedSyntheticsMonitorAttributes type
  const fullUrl = 'urls' in monitor.attributes && typeof monitor.attributes.urls === 'string' ? monitor.attributes.urls : '';
  return {
    monitor: {
      name: monitor.attributes.name,
      id: configId,
      type: monitor.attributes.type
    },
    observer: {
      geo: {
        name: ((_monitor$attributes$l = monitor.attributes.locations.find(l => l.id === locationId)) === null || _monitor$attributes$l === void 0 ? void 0 : _monitor$attributes$l.label) || ''
      }
    },
    labels: monitor.attributes.labels,
    tags: monitor.attributes.tags,
    url: {
      full: fullUrl
    },
    createdAt: monitor.created_at
  };
};
exports.getMissingPingMonitorInfo = getMissingPingMonitorInfo;
const isMonitorReadyForData = ({
  createdAt,
  monitorType
}) => {
  const waitMinutesBeforePending = monitorType === 'browser' ? 5 : 1;
  return !createdAt || createdAt && (0, _moment.default)(createdAt).isBefore((0, _moment.default)().subtract(waitMinutesBeforePending, 'minutes'));
};
const getPendingConfigs = ({
  monitorQueryIds,
  monitorLocationIds,
  upConfigs,
  downConfigs,
  monitorsData,
  monitors,
  logger
}) => {
  // Check if a config is missing, if it is it means that the monitor is pending
  const pendingConfigs = {};
  for (const monitorQueryId of monitorQueryIds) {
    for (const locationId of monitorLocationIds) {
      const configWithLocationId = `${monitorQueryId}-${locationId}`;
      const isConfigMissing = !upConfigs[configWithLocationId] && !downConfigs[configWithLocationId] && monitorsData[monitorQueryId].locations.includes(locationId);
      if (isConfigMissing) {
        const res = getMissingPingMonitorInfo({
          configId: monitorQueryId,
          locationId,
          monitors
        });
        if (res) {
          const {
            createdAt,
            ...monitorInfo
          } = res;
          if (isMonitorReadyForData({
            monitorType: monitorsData[monitorQueryId].type,
            createdAt
          })) {
            pendingConfigs[configWithLocationId] = {
              status: 'pending',
              configId: monitorQueryId,
              monitorQueryId,
              locationId,
              monitorInfo
            };
          }
        } else {
          logger.error(`Config ${configWithLocationId} not added to pending configs because the monitor info is missing`);
        }
      }
    }
  }
  return pendingConfigs;
};
exports.getPendingConfigs = getPendingConfigs;
const getConfigStats = ({
  monitorQueryIds,
  upConfigs,
  downConfigs,
  pendingConfigs
}) => {
  // Pre-organize configs by monitorId for faster lookup
  const configsByMonitor = new Map();

  // Initialize all monitors with zero counts
  for (const monitorId of monitorQueryIds) {
    configsByMonitor.set(monitorId, {
      up: 0,
      down: 0,
      pending: 0
    });
  }

  // Count up configs
  for (const configKey of Object.keys(upConfigs)) {
    const monitorId = upConfigs[configKey].monitorQueryId;
    const stats = configsByMonitor.get(monitorId);
    if (stats) stats.up++;
  }

  // Count down configs
  for (const configKey of Object.keys(downConfigs)) {
    const monitorId = downConfigs[configKey].monitorQueryId;
    const stats = configsByMonitor.get(monitorId);
    if (stats) stats.down++;
  }

  // Count pending configs
  for (const configKey of Object.keys(pendingConfigs)) {
    const monitorId = pendingConfigs[configKey].monitorQueryId;
    const stats = configsByMonitor.get(monitorId);
    if (stats) stats.pending++;
  }

  // Convert Map to the expected Record structure
  return Object.fromEntries(configsByMonitor.entries());
};
exports.getConfigStats = getConfigStats;
const calculateIsValidPing = ({
  previousRunEndTimeISO,
  scheduleInMs,
  previousRunDurationUs = 0,
  minimumTotalBufferMs = 60 * 1000 // 60 seconds
}) => {
  const msSincePreviousRunEnd = new Date().getTime() - new Date(previousRunEndTimeISO).getTime();
  const stalenessThresholdMs = scheduleInMs + Math.max(minimumTotalBufferMs, previousRunDurationUs / 1000);

  // Example: if a monitor has a schedule of 5m the last valid ping can be at (5+1)m
  // If it's greater than that it means the monitor is pending
  return msSincePreviousRunEnd < stalenessThresholdMs;
};
exports.calculateIsValidPing = calculateIsValidPing;