"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.updateFailureStoreRoute = exports.getFailureStoreStatsRoute = exports.failureStoreRoutes = void 0;
var _zod = require("@kbn/zod");
var _streamsSchema = require("@kbn/streams-schema");
var _stream_crud = require("../../../../lib/streams/stream_crud");
var _constants = require("../../../../../common/constants");
var _create_server_route = require("../../../create_server_route");
var _security_error = require("../../../../lib/streams/errors/security_error");
var _status_error = require("../../../../lib/streams/errors/status_error");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getFailureStoreStatsRoute = exports.getFailureStoreStatsRoute = (0, _create_server_route.createServerRoute)({
  endpoint: 'GET /internal/streams/{name}/failure_store/stats',
  options: {
    access: 'internal',
    summary: 'Get failure store stats',
    description: 'Gets the failure store statistics for a stream'
  },
  security: {
    authz: {
      requiredPrivileges: [_constants.STREAMS_API_PRIVILEGES.read]
    }
  },
  params: _zod.z.object({
    path: _zod.z.object({
      name: _zod.z.string()
    })
  }),
  handler: async ({
    params,
    request,
    getScopedClients,
    server
  }) => {
    const {
      scopedClusterClient,
      streamsClient
    } = await getScopedClients({
      request
    });
    const {
      name
    } = params.path;
    const [privileges, failureStore] = await Promise.all([streamsClient.getPrivileges(name), (0, _stream_crud.getFailureStore)({
      name,
      scopedClusterClient,
      isServerless: server.isServerless
    })]);
    if (!failureStore.enabled || !privileges.manage_failure_store) {
      return {
        config: failureStore,
        stats: null
      };
    }
    const stats = await (0, _stream_crud.getFailureStoreStats)({
      name,
      scopedClusterClient,
      isServerless: server.isServerless
    });
    return {
      config: failureStore,
      stats
    };
  }
});
const updateFailureStoreRoute = exports.updateFailureStoreRoute = (0, _create_server_route.createServerRoute)({
  endpoint: 'PUT /internal/streams/{name}/_failure_store',
  options: {
    access: 'internal',
    summary: 'Update failure store configuration',
    description: 'Updates the failure store configuration for a stream'
  },
  security: {
    authz: {
      requiredPrivileges: [_constants.STREAMS_API_PRIVILEGES.manage]
    }
  },
  params: _zod.z.object({
    path: _zod.z.object({
      name: _zod.z.string()
    }),
    body: _zod.z.object({
      failureStoreEnabled: _zod.z.boolean(),
      customRetentionPeriod: _zod.z.optional(_zod.z.string())
    })
  }),
  handler: async ({
    params,
    request,
    getScopedClients,
    server
  }) => {
    const {
      streamsClient,
      scopedClusterClient
    } = await getScopedClients({
      request
    });
    const {
      name
    } = params.path;
    const {
      failureStoreEnabled,
      customRetentionPeriod
    } = params.body;
    const stream = await streamsClient.getStream(name);
    if (_streamsSchema.Streams.WiredStream.Definition.is(stream)) {
      throw new _status_error.StatusError('Only wired streams can be exported', 400);
    }

    // Check if user has failure store manage privileges
    const privileges = await streamsClient.getPrivileges(name);
    if (!privileges.manage_failure_store) {
      throw new _security_error.SecurityError('Insufficient privileges to update failure store configuration');
    }
    await (0, _stream_crud.updateFailureStore)({
      name,
      enabled: failureStoreEnabled,
      customRetentionPeriod,
      scopedClusterClient,
      isServerless: server.isServerless
    });
    return {
      acknowledged: true
    };
  }
});
const failureStoreRoutes = exports.failureStoreRoutes = {
  ...getFailureStoreStatsRoute,
  ...updateFailureStoreRoute
};