"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ExecutionPlan = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _lodash = require("lodash");
var _manage_component_templates = require("../../component_templates/manage_component_templates");
var _manage_data_streams = require("../../data_streams/manage_data_streams");
var _manage_index_templates = require("../../index_templates/manage_index_templates");
var _manage_ingest_pipelines = require("../../ingest_pipelines/manage_ingest_pipelines");
var _failed_to_execute_elasticsearch_actions_error = require("../errors/failed_to_execute_elasticsearch_actions_error");
var _failed_to_plan_elasticsearch_actions_error = require("../errors/failed_to_plan_elasticsearch_actions_error");
var _insufficient_permissions_error = require("../../errors/insufficient_permissions_error");
var _required_permissions = require("./required_permissions");
var _translate_classic_stream_pipeline_actions = require("./translate_classic_stream_pipeline_actions");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */
/* eslint-disable @typescript-eslint/naming-convention */

/**
 * This class takes a list of ElasticsearchActions and groups them by type.
 * It then tries to plan these actions to make the least amount of changes to Elasticsearch resources
 * The execution of the plan aims to be as parallel as possible while still respecting the order of operations
 * to avoid any data loss
 */
class ExecutionPlan {
  constructor(dependencies) {
    (0, _defineProperty2.default)(this, "dependencies", void 0);
    (0, _defineProperty2.default)(this, "actionsByType", void 0);
    this.dependencies = dependencies;
    this.actionsByType = {
      upsert_component_template: [],
      delete_component_template: [],
      upsert_index_template: [],
      delete_index_template: [],
      upsert_ingest_pipeline: [],
      delete_ingest_pipeline: [],
      append_processor_to_ingest_pipeline: [],
      delete_processor_from_ingest_pipeline: [],
      upsert_datastream: [],
      update_lifecycle: [],
      update_default_ingest_pipeline: [],
      rollover: [],
      delete_datastream: [],
      upsert_dot_streams_document: [],
      delete_dot_streams_document: [],
      update_data_stream_mappings: [],
      delete_queries: [],
      unlink_assets: [],
      unlink_features: [],
      update_ingest_settings: []
    };
  }
  async plan(elasticsearchActions) {
    try {
      this.actionsByType = Object.assign(this.actionsByType, (0, _lodash.groupBy)(elasticsearchActions, 'type'));
      await (0, _translate_classic_stream_pipeline_actions.translateClassicStreamPipelineActions)(this.actionsByType, this.dependencies.scopedClusterClient);
    } catch (error) {
      throw new _failed_to_plan_elasticsearch_actions_error.FailedToPlanElasticsearchActionsError(`Failed to plan Elasticsearch action execution: ${error.message}`);
    }
    await this.validatePermissions();
  }
  plannedActions() {
    return this.actionsByType;
  }
  async validatePermissions() {
    const {
      actionsByType
    } = this;
    const requiredPermissions = (0, _required_permissions.getRequiredPermissionsForActions)({
      actionsByType,
      isServerless: this.dependencies.isServerless
    });

    // Check if we have any permissions to validate
    if (requiredPermissions.cluster.length === 0 && Object.keys(requiredPermissions.index).length === 0) {
      return true;
    }

    // Use security API to check if user has all required permissions
    const securityClient = this.dependencies.scopedClusterClient.asCurrentUser.security;
    const hasPrivilegesRequest = {
      cluster: requiredPermissions.cluster.length > 0 ? requiredPermissions.cluster : undefined
    };

    // Add index privileges if there are any
    if (Object.keys(requiredPermissions.index).length > 0) {
      hasPrivilegesRequest.index = Object.entries(requiredPermissions.index).map(([index, privileges]) => ({
        names: [index],
        privileges
      }));
    }
    const hasPrivilegesResponse = await securityClient.hasPrivileges(hasPrivilegesRequest);
    if (!hasPrivilegesResponse.has_all_requested) {
      throw new _insufficient_permissions_error.InsufficientPermissionsError('User does not have sufficient permissions to execute these actions', hasPrivilegesResponse);
    }
  }
  async execute() {
    try {
      const {
        upsert_component_template,
        delete_component_template,
        upsert_index_template,
        delete_index_template,
        upsert_ingest_pipeline,
        delete_ingest_pipeline,
        append_processor_to_ingest_pipeline,
        delete_processor_from_ingest_pipeline,
        upsert_datastream,
        update_lifecycle,
        rollover,
        update_default_ingest_pipeline,
        delete_datastream,
        upsert_dot_streams_document,
        delete_dot_streams_document,
        update_data_stream_mappings,
        delete_queries,
        unlink_assets,
        unlink_features,
        update_ingest_settings,
        ...rest
      } = this.actionsByType;
      assertEmptyObject(rest);
      if (append_processor_to_ingest_pipeline.length !== 0) {
        throw new Error('append_processor_to_ingest_pipeline actions have not been translated');
      }
      if (delete_processor_from_ingest_pipeline.length !== 0) {
        throw new Error('delete_processor_from_ingest_pipeline actions have not been translated');
      }

      // This graph is parallelizing as much as possible
      // It's important we don't make changes too early, otherwise things can break halfway through
      // such as leading to data loss if routing changes too early

      await Promise.all([this.upsertComponentTemplates(upsert_component_template), this.upsertIndexTemplates(upsert_index_template)]);
      await this.upsertDatastreams(upsert_datastream);
      await this.updateIngestSettings(update_ingest_settings);
      await Promise.all([this.rollover(rollover), this.updateLifecycle(update_lifecycle), this.updateDataStreamMappingsAndRollover(update_data_stream_mappings), this.updateDefaultIngestPipeline(update_default_ingest_pipeline)]);
      await this.upsertIngestPipelines(upsert_ingest_pipeline);
      await this.deleteDatastreams(delete_datastream);
      await this.deleteIndexTemplates(delete_index_template);
      await Promise.all([this.deleteComponentTemplates(delete_component_template), this.deleteIngestPipelines(delete_ingest_pipeline), this.deleteQueries(delete_queries), this.unlinkAssets(unlink_assets), this.unlinkFeatures(unlink_features)]);
      await this.upsertAndDeleteDotStreamsDocuments([...upsert_dot_streams_document, ...delete_dot_streams_document]);
    } catch (error) {
      throw new _failed_to_execute_elasticsearch_actions_error.FailedToExecuteElasticsearchActionsError(`Failed to execute Elasticsearch actions: ${error.message}`);
    }
  }
  async deleteQueries(actions) {
    if (actions.length === 0) {
      return;
    }
    return Promise.all(actions.map(action => this.dependencies.queryClient.deleteAll(action.request.name)));
  }
  async unlinkAssets(actions) {
    if (actions.length === 0) {
      return;
    }
    return Promise.all(actions.flatMap(action => [this.dependencies.assetClient.syncAssetList(action.request.name, [], 'dashboard'), this.dependencies.assetClient.syncAssetList(action.request.name, [], 'rule'), this.dependencies.assetClient.syncAssetList(action.request.name, [], 'slo')]));
  }
  async unlinkFeatures(actions) {
    if (actions.length === 0) {
      return;
    }
    return Promise.all(actions.map(action => this.dependencies.featureClient.syncFeatureList(action.request.name, [])));
  }
  async upsertComponentTemplates(actions) {
    return Promise.all(actions.map(action => (0, _manage_component_templates.upsertComponent)({
      esClient: this.dependencies.scopedClusterClient.asCurrentUser,
      logger: this.dependencies.logger,
      component: action.request
    })));
  }
  async upsertIndexTemplates(actions) {
    return Promise.all(actions.map(action => (0, _manage_index_templates.upsertTemplate)({
      esClient: this.dependencies.scopedClusterClient.asCurrentUser,
      logger: this.dependencies.logger,
      template: action.request
    })));
  }
  async rollover(actions) {
    return Promise.all(actions.map(action => (0, _manage_data_streams.rolloverDataStream)({
      esClient: this.dependencies.scopedClusterClient.asCurrentUser,
      logger: this.dependencies.logger,
      name: action.request.name
    })));
  }
  async updateDefaultIngestPipeline(actions) {
    return Promise.all(actions.map(action => (0, _manage_data_streams.updateDefaultIngestPipeline)({
      esClient: this.dependencies.scopedClusterClient.asCurrentUser,
      name: action.request.name,
      pipeline: action.request.pipeline
    })));
  }
  async updateLifecycle(actions) {
    return Promise.all(actions.map(action => (0, _manage_data_streams.updateDataStreamsLifecycle)({
      esClient: this.dependencies.scopedClusterClient.asCurrentUser,
      logger: this.dependencies.logger,
      names: [action.request.name],
      lifecycle: action.request.lifecycle,
      isServerless: this.dependencies.isServerless
    })));
  }
  async updateDataStreamMappingsAndRollover(actions) {
    return Promise.all(actions.map(action => (0, _manage_data_streams.updateDataStreamsMappings)({
      esClient: this.dependencies.scopedClusterClient.asCurrentUser,
      logger: this.dependencies.logger,
      name: action.request.name,
      mappings: action.request.mappings
    })));
  }
  async upsertDatastreams(actions) {
    return Promise.all(actions.map(action => (0, _manage_data_streams.upsertDataStream)({
      esClient: this.dependencies.scopedClusterClient.asCurrentUser,
      logger: this.dependencies.logger,
      name: action.request.name
    })));
  }
  async upsertIngestPipelines(actions) {
    const actionWithStreamsDepth = actions.map(action => {
      var _action$stream$match$, _action$stream$match;
      return {
        ...action,
        depth: (_action$stream$match$ = (_action$stream$match = action.stream.match(/\./g)) === null || _action$stream$match === void 0 ? void 0 : _action$stream$match.length) !== null && _action$stream$match$ !== void 0 ? _action$stream$match$ : 0
      };
    });
    return Promise.all((0, _lodash.orderBy)(actionWithStreamsDepth, 'depth', 'desc').map(action => (0, _manage_ingest_pipelines.upsertIngestPipeline)({
      esClient: this.dependencies.scopedClusterClient.asCurrentUser,
      logger: this.dependencies.logger,
      pipeline: action.request
    })));
  }
  async deleteDatastreams(actions) {
    return Promise.all(actions.map(action => (0, _manage_data_streams.deleteDataStream)({
      esClient: this.dependencies.scopedClusterClient.asCurrentUser,
      logger: this.dependencies.logger,
      name: action.request.name
    })));
  }
  async deleteIndexTemplates(actions) {
    return Promise.all(actions.map(action => (0, _manage_index_templates.deleteTemplate)({
      esClient: this.dependencies.scopedClusterClient.asCurrentUser,
      logger: this.dependencies.logger,
      name: action.request.name
    })));
  }
  async deleteIngestPipelines(actions) {
    return Promise.all(actions.map(action => (0, _manage_ingest_pipelines.deleteIngestPipeline)({
      esClient: this.dependencies.scopedClusterClient.asCurrentUser,
      logger: this.dependencies.logger,
      id: action.request.name
    })));
  }
  async deleteComponentTemplates(actions) {
    return Promise.all(actions.map(action => (0, _manage_component_templates.deleteComponent)({
      esClient: this.dependencies.scopedClusterClient.asCurrentUser,
      logger: this.dependencies.logger,
      name: action.request.name
    })));
  }
  async upsertAndDeleteDotStreamsDocuments(actions) {
    return this.dependencies.storageClient.bulk({
      operations: actions.map(dotDocumentActionToBulkOperation),
      refresh: true
    });
  }
  async updateIngestSettings(actions) {
    return Promise.all(actions.map(action => (0, _manage_data_streams.putDataStreamsSettings)({
      esClient: this.dependencies.scopedClusterClient.asCurrentUser,
      names: [action.request.name],
      settings: action.request.settings
    })));
  }
}
exports.ExecutionPlan = ExecutionPlan;
function dotDocumentActionToBulkOperation(action) {
  if (action.type === 'upsert_dot_streams_document') {
    return {
      index: {
        document: action.request,
        _id: action.request.name
      }
    };
  }
  return {
    delete: {
      _id: action.request.name
    }
  };
}
function assertEmptyObject(object) {
  // This is for type checking only
}