"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getConnectorType = exports.configValidator = void 0;
var _i18n = require("@kbn/i18n");
var _types = require("@kbn/actions-plugin/server/sub_action_framework/types");
var _common = require("@kbn/actions-plugin/common");
var _constants = require("../../../common/inference/constants");
var _schema = require("../../../common/inference/schema");
var _inference = require("./inference");
var _unflatten_object = require("../lib/unflatten_object");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const deleteInferenceEndpoint = async (inferenceId, taskType, logger, esClient) => {
  try {
    await esClient.inference.delete({
      task_type: taskType,
      inference_id: inferenceId
    });
    logger.debug(`Inference endpoint for task type "${taskType}" and inference id ${inferenceId} was successfuly deleted`);
  } catch (e) {
    logger.warn(`Failed to delete inference endpoint for task type "${taskType}" and inference id ${inferenceId}. Error: ${e.message}`);
    throw e;
  }
};
const getConnectorType = () => ({
  id: _constants.INFERENCE_CONNECTOR_ID,
  name: _constants.INFERENCE_CONNECTOR_TITLE,
  getService: params => new _inference.InferenceConnector(params),
  schema: {
    config: _schema.ConfigSchema,
    secrets: _schema.SecretsSchema
  },
  validators: [{
    type: _types.ValidatorType.CONFIG,
    validator: configValidator
  }],
  supportedFeatureIds: [_common.GenerativeAIForSecurityConnectorFeatureId, _common.GenerativeAIForSearchPlaygroundConnectorFeatureId, _common.GenerativeAIForObservabilityConnectorFeatureId],
  minimumLicenseRequired: 'enterprise',
  preSaveHook: async ({
    config,
    secrets,
    logger,
    services,
    isUpdate
  }) => {
    const esClient = services.scopedClusterClient.asInternalUser;
    try {
      var _config$providerConfi, _secrets$providerSecr, _config$inferenceId;
      const taskSettings = config !== null && config !== void 0 && config.taskTypeConfig ? {
        ...(0, _unflatten_object.unflattenObject)(config === null || config === void 0 ? void 0 : config.taskTypeConfig)
      } : {};
      const serviceSettings = {
        ...(0, _unflatten_object.unflattenObject)((_config$providerConfi = config === null || config === void 0 ? void 0 : config.providerConfig) !== null && _config$providerConfi !== void 0 ? _config$providerConfi : {}),
        ...(0, _unflatten_object.unflattenObject)((_secrets$providerSecr = secrets === null || secrets === void 0 ? void 0 : secrets.providerSecrets) !== null && _secrets$providerSecr !== void 0 ? _secrets$providerSecr : {})
      };
      let inferenceExists = false;
      try {
        await (esClient === null || esClient === void 0 ? void 0 : esClient.inference.get({
          inference_id: config === null || config === void 0 ? void 0 : config.inferenceId,
          task_type: config === null || config === void 0 ? void 0 : config.taskType
        }));
        inferenceExists = true;
      } catch (e) {
        /* throws error if inference endpoint by id does not exist */
      }
      if (!isUpdate && inferenceExists) {
        throw new Error(`Inference with id ${config === null || config === void 0 ? void 0 : config.inferenceId} and task type ${config === null || config === void 0 ? void 0 : config.taskType} already exists.`);
      }
      if (isUpdate && inferenceExists && config && config.provider) {
        // TODO: replace, when update API for inference endpoint exists
        await deleteInferenceEndpoint(config.inferenceId, config.taskType, logger, esClient);
      }
      await (esClient === null || esClient === void 0 ? void 0 : esClient.inference.put({
        inference_id: (_config$inferenceId = config === null || config === void 0 ? void 0 : config.inferenceId) !== null && _config$inferenceId !== void 0 ? _config$inferenceId : '',
        task_type: config === null || config === void 0 ? void 0 : config.taskType,
        inference_config: {
          service: config.provider,
          service_settings: serviceSettings,
          task_settings: taskSettings
        }
      }));
      logger.debug(`Inference endpoint for task type "${config === null || config === void 0 ? void 0 : config.taskType}" and inference id ${config === null || config === void 0 ? void 0 : config.inferenceId} was successfuly ${isUpdate ? 'updated' : 'created'}`);
    } catch (e) {
      logger.warn(`Failed to ${isUpdate ? 'update' : 'create'} inference endpoint for task type "${config === null || config === void 0 ? void 0 : config.taskType}" and inference id ${config === null || config === void 0 ? void 0 : config.inferenceId}. Error: ${e.message}`);
      throw e;
    }
  },
  postSaveHook: async ({
    config,
    logger,
    services,
    wasSuccessful,
    isUpdate
  }) => {
    if (!wasSuccessful && !isUpdate) {
      const esClient = services.scopedClusterClient.asInternalUser;
      await deleteInferenceEndpoint(config.inferenceId, config.taskType, logger, esClient);
    }
  },
  postDeleteHook: async ({
    config,
    logger,
    services
  }) => {
    const esClient = services.scopedClusterClient.asInternalUser;
    await deleteInferenceEndpoint(config.inferenceId, config.taskType, logger, esClient);
  }
});
exports.getConnectorType = getConnectorType;
const configValidator = (configObject, validatorServices) => {
  try {
    const {
      provider,
      taskType
    } = configObject;
    if (!Object.keys(_constants.ServiceProviderKeys).includes(provider)) {
      throw new Error(`API Provider is not supported${provider && provider.length ? `: ${provider}` : ``} by Inference Endpoint.`);
    }
    if (taskType === 'chat_completion' && !Object.keys(_constants.SUB_ACTION).includes('UNIFIED_COMPLETION')) {
      throw new Error(`Task type is not supported${taskType && taskType.length ? `: ${taskType}` : ``} by Inference Endpoint.`);
    }
    if (taskType !== 'chat_completion' && !Object.keys(_constants.SUB_ACTION).includes(taskType.toUpperCase())) {
      throw new Error(`Task type is not supported${taskType && taskType.length ? `: ${taskType}` : ``} by Inference Endpoint.`);
    }
    return configObject;
  } catch (err) {
    throw new Error(_i18n.i18n.translate('xpack.stackConnectors.inference.configurationErrorApiProvider', {
      defaultMessage: 'Error configuring Inference API action: {err}',
      values: {
        err: err.toString()
      }
    }));
  }
};
exports.configValidator = configValidator;