"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.summarizeCopyResult = summarizeCopyResult;
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const isAnyConflict = failure => failure.error.type === 'conflict' || failure.error.type === 'ambiguous_conflict';
const isMissingReferences = failure => failure.error.type === 'missing_references';
const isUnresolvableError = failure => !isAnyConflict(failure) && !isMissingReferences(failure);
const typeComparator = (a, b) => a.type > b.type ? 1 : a.type < b.type ? -1 : 0;
function summarizeCopyResult(savedObjectTarget, copyResult) {
  var _copyResult$failedImp, _copyResult$failedImp2, _copyResult$failedImp3, _copyResult$failedImp4, _copyResult$successfu;
  const conflicts = (_copyResult$failedImp = copyResult === null || copyResult === void 0 ? void 0 : copyResult.failedImports.filter(isAnyConflict)) !== null && _copyResult$failedImp !== void 0 ? _copyResult$failedImp : [];
  const missingReferences = (_copyResult$failedImp2 = copyResult === null || copyResult === void 0 ? void 0 : copyResult.failedImports.filter(isMissingReferences)) !== null && _copyResult$failedImp2 !== void 0 ? _copyResult$failedImp2 : [];
  const unresolvableErrors = (_copyResult$failedImp3 = copyResult === null || copyResult === void 0 ? void 0 : copyResult.failedImports.filter(failed => isUnresolvableError(failed))) !== null && _copyResult$failedImp3 !== void 0 ? _copyResult$failedImp3 : [];
  const getExtraFields = ({
    type,
    id
  }) => {
    const conflict = conflicts.find(({
      obj
    }) => obj.type === type && obj.id === id);
    const missingReference = missingReferences.find(({
      obj
    }) => obj.type === type && obj.id === id);
    const hasMissingReferences = missingReference !== undefined;
    const hasUnresolvableErrors = unresolvableErrors.some(({
      obj
    }) => obj.type === type && obj.id === id);
    const overwrite = conflict ? false : missingReference ? missingReference.obj.overwrite === true : (copyResult === null || copyResult === void 0 ? void 0 : copyResult.successfulImports.some(obj => obj.type === type && obj.id === id && obj.overwrite)) === true;
    return {
      conflict,
      hasMissingReferences,
      hasUnresolvableErrors,
      overwrite
    };
  };
  const objectMap = new Map();
  objectMap.set(`${savedObjectTarget.type}:${savedObjectTarget.id}`, {
    type: savedObjectTarget.type,
    id: savedObjectTarget.id,
    name: savedObjectTarget.title,
    icon: savedObjectTarget.icon,
    ...getExtraFields(savedObjectTarget)
  });
  const addObjectsToMap = objects => {
    objects.forEach(obj => {
      const {
        type,
        id,
        meta
      } = obj;
      objectMap.set(`${type}:${id}`, {
        type,
        id,
        name: meta.title || `${type} [id=${id}]`,
        icon: meta.icon || 'apps',
        ...getExtraFields(obj)
      });
    });
  };
  const failedImports = ((_copyResult$failedImp4 = copyResult === null || copyResult === void 0 ? void 0 : copyResult.failedImports) !== null && _copyResult$failedImp4 !== void 0 ? _copyResult$failedImp4 : []).map(({
    obj
  }) => obj).sort(typeComparator);
  addObjectsToMap(failedImports);
  const successfulImports = ((_copyResult$successfu = copyResult === null || copyResult === void 0 ? void 0 : copyResult.successfulImports) !== null && _copyResult$successfu !== void 0 ? _copyResult$successfu : []).sort(typeComparator);
  addObjectsToMap(successfulImports);
  if (typeof copyResult === 'undefined') {
    return {
      processing: true,
      objects: Array.from(objectMap.values())
    };
  }
  const successful = Boolean(copyResult && copyResult.failedImports.length === 0);
  if (successful) {
    return {
      successful,
      hasConflicts: false,
      objects: Array.from(objectMap.values()),
      hasMissingReferences: false,
      hasUnresolvableErrors: false,
      processing: false
    };
  }
  const hasConflicts = conflicts.length > 0;
  const hasMissingReferences = missingReferences.length > 0;
  const hasUnresolvableErrors = unresolvableErrors.length > 0;
  return {
    successful,
    hasConflicts,
    objects: Array.from(objectMap.values()),
    hasMissingReferences,
    hasUnresolvableErrors,
    processing: false
  };
}