"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ShareRegistry = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

class ShareRegistry {
  constructor() {
    (0, _defineProperty2.default)(this, "urlService", void 0);
    (0, _defineProperty2.default)(this, "anonymousAccessServiceProvider", void 0);
    (0, _defineProperty2.default)(this, "capabilities", void 0);
    (0, _defineProperty2.default)(this, "getLicense", void 0);
    (0, _defineProperty2.default)(this, "globalMarker", '*');
    (0, _defineProperty2.default)(this, "shareOptionsStore", {
      [this.globalMarker]: new Map()
    });
    // register default share actions
    this.registerLinkShareAction();
    this.registerEmbedShareAction();
  }
  setup() {
    return {
      /**
       * @deprecated Use {@link registerShareIntegration} instead.
       */
      register: this.register.bind(this),
      registerShareIntegration: this.registerShareIntegration.bind(this)
    };
  }
  start({
    urlService,
    anonymousAccessServiceProvider,
    capabilities,
    getLicense
  }) {
    this.urlService = urlService;
    this.anonymousAccessServiceProvider = anonymousAccessServiceProvider;
    this.capabilities = capabilities;
    this.getLicense = getLicense;
    return {
      availableIntegrations: this.availableIntegrations.bind(this),
      resolveShareItemsForShareContext: this.resolveShareItemsForShareContext.bind(this)
    };
  }
  registerShareIntentAction(shareObject, shareActionIntent) {
    if (!this.shareOptionsStore[shareObject]) {
      this.shareOptionsStore[shareObject] = new Map();
    }
    const shareContextMap = this.shareOptionsStore[shareObject];
    const recordKey = shareActionIntent.shareType === 'integration' ? `integration-${shareActionIntent.groupId || 'unknown'}-${shareActionIntent.id}` : shareActionIntent.shareType;
    if (shareContextMap.has(recordKey)) {
      throw new Error(`Share action with type [${shareActionIntent.shareType}] for app [${shareObject}] has already been registered.`);
    }
    shareContextMap.set(recordKey, shareActionIntent);
  }
  registerLinkShareAction() {
    this.registerShareIntentAction(this.globalMarker, {
      shareType: 'link',
      config: ({
        urlService
      }) => ({
        shortUrlService: urlService === null || urlService === void 0 ? void 0 : urlService.shortUrls.get(null)
      })
    });
  }
  registerEmbedShareAction() {
    this.registerShareIntentAction(this.globalMarker, {
      shareType: 'embed',
      config: ({
        urlService,
        anonymousAccessServiceProvider
      }) => ({
        anonymousAccess: anonymousAccessServiceProvider(),
        shortUrlService: urlService.shortUrls.get(null)
      })
    });
  }

  /**
   * @description provides an escape hatch to support allowing legacy share menu items to be registered
   */
  register(value) {
    // implement backwards compatibility for the share plugin
    this.registerShareIntentAction(this.globalMarker, {
      shareType: 'legacy',
      id: value.id,
      config: value.getShareMenuItemsLegacy
    });
  }
  registerShareIntegration(...args) {
    const [shareObject, shareActionIntent] = args.length === 1 ? [this.globalMarker, args[0]] : args;
    this.registerShareIntentAction(shareObject, {
      shareType: 'integration',
      id: shareActionIntent.id,
      groupId: shareActionIntent.groupId,
      config: shareActionIntent.getShareIntegrationConfig,
      prerequisiteCheck: shareActionIntent.prerequisiteCheck
    });
  }
  getShareConfigOptionsForObject(objectType) {
    const shareContextMap = this.shareOptionsStore[objectType];
    const globalOptions = Array.from(this.shareOptionsStore[this.globalMarker].values());
    if (!shareContextMap) {
      return globalOptions;
    }
    return globalOptions.concat(Array.from(shareContextMap.values()));
  }

  /**
   * Returns all share actions that are available for the given object type.
   */
  availableIntegrations(objectType, groupId) {
    if (!this.capabilities || !this.getLicense) {
      throw new Error('ShareOptionsManager#start was not invoked');
    }
    return this.getShareConfigOptionsForObject(objectType).filter(share => {
      if (groupId && (share.shareType !== 'integration' || (share === null || share === void 0 ? void 0 : share.groupId) !== groupId && share.shareType === 'integration')) {
        return false;
      }
      if (share.shareType === 'integration' && share.prerequisiteCheck) {
        return share.prerequisiteCheck({
          capabilities: this.capabilities,
          license: this.getLicense(),
          objectType
        });
      }

      // if no activation requirement is provided, assume that the share action is always available
      return true;
    });
  }
  async resolveShareItemsForShareContext({
    objectType,
    isServerless,
    ...shareContext
  }) {
    if (!this.urlService || !this.anonymousAccessServiceProvider) {
      throw new Error('ShareOptionsManager#start was not invoked');
    }
    return (await Promise.all(this.availableIntegrations(objectType).map(async shareAction => {
      let config;
      if (shareAction.shareType === 'legacy') {
        config = shareAction.config.call(null, {
          objectType,
          ...shareContext
        });
      } else if (shareAction.shareType === 'integration') {
        config = await shareAction.config.call(null, {
          urlService: this.urlService,
          anonymousAccessServiceProvider: this.anonymousAccessServiceProvider,
          objectType,
          ...shareContext
        });
      } else {
        config = shareAction.config.call(null, {
          urlService: this.urlService,
          anonymousAccessServiceProvider: this.anonymousAccessServiceProvider,
          objectType,
          ...shareContext
        });
      }
      return {
        ...shareAction,
        config
      };
    }))).filter(shareAction => {
      return isServerless ? shareAction.shareType !== 'embed' && shareAction.config : shareAction.config;
    });
  }
}
exports.ShareRegistry = ShareRegistry;