"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.SecurityTelemetryTask = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _moment = _interopRequireDefault(require("moment"));
var _task_state = require("./task_state");
var _helpers = require("./helpers");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class SecurityTelemetryTask {
  constructor(config, logger, sender, receiver, taskMetricsService) {
    (0, _defineProperty2.default)(this, "config", void 0);
    (0, _defineProperty2.default)(this, "logger", void 0);
    (0, _defineProperty2.default)(this, "sender", void 0);
    (0, _defineProperty2.default)(this, "receiver", void 0);
    (0, _defineProperty2.default)(this, "taskMetricsService", void 0);
    (0, _defineProperty2.default)(this, "getLastExecutionTime", (taskExecutionTime, taskInstance) => {
      var _taskInstance$state;
      return this.config.getLastExecutionTime ? this.config.getLastExecutionTime(taskExecutionTime, (_taskInstance$state = taskInstance.state) === null || _taskInstance$state === void 0 ? void 0 : _taskInstance$state.lastExecutionTimestamp) : undefined;
    });
    (0, _defineProperty2.default)(this, "getTaskId", () => {
      return `${this.config.type}:${this.config.version}`;
    });
    (0, _defineProperty2.default)(this, "register", taskManager => {
      taskManager.registerTaskDefinitions({
        [this.config.type]: {
          title: this.config.title,
          timeout: this.config.timeout,
          stateSchemaByVersion: _task_state.stateSchemaByVersion,
          createTaskRunner: ({
            taskInstance
          }) => {
            const state = taskInstance.state;
            return {
              run: async () => {
                const taskExecutionTime = (0, _moment.default)().utc().toISOString();
                const executionPeriod = {
                  last: this.getLastExecutionTime(taskExecutionTime, taskInstance),
                  current: taskExecutionTime
                };
                const hits = await this.runTask(taskInstance.id, executionPeriod);
                const updatedState = {
                  lastExecutionTimestamp: taskExecutionTime,
                  runs: state.runs + 1,
                  hits
                };
                return {
                  state: updatedState
                };
              },
              cancel: async () => {}
            };
          }
        }
      });
    });
    (0, _defineProperty2.default)(this, "start", async taskManager => {
      const taskId = this.getTaskId();
      this.logger.debug('Attempting to schedule task', {
        taskId
      });
      try {
        await taskManager.ensureScheduled({
          id: taskId,
          taskType: this.config.type,
          scope: ['securitySolution'],
          schedule: {
            interval: this.config.interval
          },
          state: _task_state.emptyState,
          params: {
            version: this.config.version
          }
        });
      } catch (error) {
        this.logger.error('Error scheduling task', {
          error
        });
      }
    });
    (0, _defineProperty2.default)(this, "runTask", async (taskId, executionPeriod) => {
      this.logger.debug('Attempting to run', {
        taskId
      });
      if (taskId !== this.getTaskId()) {
        this.logger.info('outdated task', {
          taskId
        });
        return 0;
      }
      const isOptedIn = await this.sender.isTelemetryOptedIn();
      if (!isOptedIn) {
        this.logger.info('Telemetry is not opted-in', {
          taskId
        });
        return 0;
      }
      const isTelemetryServicesReachable = await this.sender.isTelemetryServicesReachable();
      if (!isTelemetryServicesReachable) {
        this.logger.info('Cannot reach telemetry services', {
          taskId
        });
        return 0;
      }
      this.logger.debug('Running task', {
        taskId
      });
      return this.config.runTask(taskId, this.logger, this.receiver, this.sender, this.taskMetricsService, executionPeriod);
    });
    this.config = config;
    this.logger = (0, _helpers.newTelemetryLogger)(logger.get('task'));
    this.sender = sender;
    this.receiver = receiver;
    this.taskMetricsService = taskMetricsService;
  }
}
exports.SecurityTelemetryTask = SecurityTelemetryTask;