"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.SiemMigrationsBaseEvaluator = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _evaluation = require("langsmith/evaluation");
var _langsmith = require("@kbn/langchain/server/tracers/langsmith");
var _langsmith2 = require("langsmith");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class SiemMigrationsBaseEvaluator {
  /**
   * Custom evaluators are the functions that will be used to evaluate the quality if the results  in langchain.
   * Both Rule and Dashboard migrations can have common or differing evaluators.
   */

  constructor(taskRunner, dependencies, logger) {
    (0, _defineProperty2.default)(this, "genericEvaluators", {
      translation_result: ({
        run,
        example
      }) => {
        var _run$outputs, _example$outputs;
        const runResult = run === null || run === void 0 ? void 0 : (_run$outputs = run.outputs) === null || _run$outputs === void 0 ? void 0 : _run$outputs.translation_result;
        const expectedResult = example === null || example === void 0 ? void 0 : (_example$outputs = example.outputs) === null || _example$outputs === void 0 ? void 0 : _example$outputs.translation_result;
        if (!expectedResult) {
          return {
            comment: 'No translation result expected'
          };
        }
        if (!runResult) {
          return {
            score: false,
            comment: 'No translation result received'
          };
        }
        if (runResult === expectedResult) {
          return {
            score: true,
            comment: 'Correct'
          };
        }
        return {
          score: false,
          comment: `Incorrect, expected "${expectedResult}" but got "${runResult}"`
        };
      }
    });
    this.taskRunner = taskRunner;
    this.dependencies = dependencies;
    this.logger = logger;
  }
  async evaluate({
    connectorId,
    langsmithOptions
  }) {
    if (!(0, _langsmith.isLangSmithEnabled)()) {
      throw Error('LangSmith is not enabled');
    }
    const client = new _langsmith2.Client({
      apiKey: langsmithOptions.api_key
    });

    // Make sure the dataset exists
    const dataset = [];
    for await (const example of client.listExamples({
      datasetName: langsmithOptions.dataset
    })) {
      dataset.push(example);
    }
    if (dataset.length === 0) {
      throw Error(`LangSmith dataset not found: ${langsmithOptions.dataset}`);
    }

    // Initialize the task runner first, this may take some time
    await this.taskRunner.initialize();

    // Check if the connector exists and user has privileges to read it
    const connector = await this.dependencies.actionsClient.get({
      id: connectorId
    });
    if (!connector) {
      throw Error(`Connector with id ${connectorId} not found`);
    }
    // for each connector, setup the evaluator
    await this.taskRunner.setup(connectorId);

    // create the migration task after setup
    const evaluators = this.getEvaluators();
    const executeMigrationTask = (params, config) => this.taskRunner.executeTask(params, config);
    (0, _evaluation.evaluate)(executeMigrationTask, {
      data: langsmithOptions.dataset,
      experimentPrefix: connector.name,
      evaluators,
      client,
      maxConcurrency: 3
    }).then(() => {
      this.logger.info('Evaluation finished');
    }).catch(err => {
      this.logger.error(`Evaluation error:\n ${JSON.stringify(err, null, 2)}`);
    });
  }
  getEvaluators() {
    return Object.entries({
      ...this.genericEvaluators,
      ...this.evaluators
    }).map(([key, evaluator]) => {
      return args => {
        const result = evaluator(args);
        return {
          key,
          ...result
        };
      };
    });
  }
}
exports.SiemMigrationsBaseEvaluator = SiemMigrationsBaseEvaluator;