"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createEventLogReader = void 0;
var _securitysolutionRules = require("@kbn/securitysolution-rules");
var _rule_monitoring = require("../../../../../../../common/api/detection_engine/rule_monitoring");
var _execution_log = require("../../../../../../../common/detection_engine/rule_management/execution_log");
var _kql = require("../../../../../../../common/utils/kql");
var _utility_types = require("../../../../../../../common/utility_types");
var _invariant = require("../../../../../../../common/utils/invariant");
var _with_security_span = require("../../../../../../utils/with_security_span");
var _kql2 = require("../../utils/kql");
var _execution_results = require("./aggregations/execution_results");
var _event_log_constants = require("../../event_log/event_log_constants");
var f = _interopRequireWildcard(require("../../event_log/event_log_fields"));
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && {}.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const createEventLogReader = eventLog => {
  return {
    async getExecutionEvents(args) {
      const {
        ruleId,
        searchTerm,
        eventTypes,
        logLevels,
        dateStart,
        dateEnd,
        sortOrder,
        page,
        perPage
      } = args;
      const soType = _event_log_constants.RULE_SAVED_OBJECT_TYPE;
      const soIds = [ruleId];
      const findResult = await (0, _with_security_span.withSecuritySpan)('findEventsBySavedObjectIds', () => {
        return eventLog.findEventsBySavedObjectIds(soType, soIds, {
          // TODO: include Framework events
          filter: buildEventLogKqlFilter({
            searchTerm,
            eventTypes,
            logLevels,
            dateStart,
            dateEnd
          }),
          sort: [{
            sort_field: f.TIMESTAMP,
            sort_order: sortOrder
          }, {
            sort_field: f.EVENT_SEQUENCE,
            sort_order: sortOrder
          }],
          page,
          per_page: perPage
        });
      });
      return {
        events: findResult.data.map(event => normalizeEvent(event)),
        pagination: {
          page: findResult.page,
          per_page: findResult.per_page,
          total: findResult.total
        }
      };
    },
    async getExecutionResults(args) {
      const {
        ruleId,
        start,
        end,
        statusFilters,
        page,
        perPage,
        sortField,
        sortOrder,
        runTypeFilters
      } = args;
      const soType = _event_log_constants.RULE_SAVED_OBJECT_TYPE;
      const soIds = [ruleId];
      let totalExecutions;
      let executionIdsFilter = '';

      // Similar workaround to the above for status filters
      // First fetch execution uuid's by run type filter if provided
      // Then use those ID's to filter by status if provided
      const MAX_RUN_TYPE_FILTERS = _execution_log.RUN_TYPE_FILTERS.length;
      const someRunTypeFiltersSelected = runTypeFilters.length > 0 && runTypeFilters.length < MAX_RUN_TYPE_FILTERS;
      if (someRunTypeFiltersSelected) {
        const ruleRunEventActions = runTypeFilters.map(runType => {
          return {
            [_rule_monitoring.RuleRunTypeEnum.standard]: 'execute',
            [_rule_monitoring.RuleRunTypeEnum.backfill]: 'execute-backfill'
          }[runType];
        });
        const {
          executionIds,
          total
        } = await findRuleExecutionIds({
          eventLog,
          soType,
          soIds,
          start,
          end,
          filter: `event.provider:alerting AND (${ruleRunEventActions.map(runType => `event.action:${runType}`).join(' OR ')}) `
        });
        if (executionIds.length === 0) {
          return {
            total: 0,
            events: []
          };
        } else {
          totalExecutions = total;
          executionIdsFilter = `${f.RULE_EXECUTION_UUID}:(${executionIds.join(' OR ')})`;
        }
      }

      // Current workaround to support root level filters without missing fields in the aggregate event
      // or including events from statuses that aren't selected
      // TODO: See: https://github.com/elastic/kibana/pull/127339/files#r825240516
      // First fetch execution uuid's by status filter if provided
      // If 0 or 3 statuses are selected we can search for all statuses and don't need this pre-filter by ID
      const MAX_STATUSES_FILTERS = _execution_log.STATUS_FILTERS.length;
      const someStatusFiltersSelected = statusFilters.length > 0 && statusFilters.length < MAX_STATUSES_FILTERS;
      if (someStatusFiltersSelected) {
        const outcomes = (0, _execution_results.mapRuleExecutionStatusToPlatformStatus)(statusFilters);
        const outcomeFilter = outcomes.length ? `OR event.outcome:(${outcomes.join(' OR ')})` : '';
        const {
          executionIds,
          total
        } = await findRuleExecutionIds({
          eventLog,
          soType,
          soIds,
          start,
          end,
          filter: `(${f.RULE_EXECUTION_STATUS}:(${statusFilters.join(' OR ')}) ${outcomeFilter}) ${executionIdsFilter ? `AND ${executionIdsFilter}` : ''}`
        });

        // Early return if no results based on status filter
        if (executionIds.length === 0) {
          return {
            total: 0,
            events: []
          };
        } else {
          executionIdsFilter = `${f.RULE_EXECUTION_UUID}:(${executionIds.join(' OR ')})`;
          totalExecutions = total;
        }
      }

      // Now query for aggregate events, and pass any ID's as filters as determined from the above status/queryText results
      const results = await eventLog.aggregateEventsBySavedObjectIds(soType, soIds, {
        start,
        end,
        filter: executionIdsFilter,
        aggs: (0, _execution_results.getExecutionEventAggregation)({
          maxExecutions: _securitysolutionRules.MAX_EXECUTION_EVENTS_DISPLAYED,
          page,
          perPage,
          sort: [{
            [sortField]: {
              order: sortOrder
            }
          }],
          runTypeFilters
        })
      });
      return (0, _execution_results.formatExecutionEventResponse)(results, totalExecutions);
    }
  };
};
exports.createEventLogReader = createEventLogReader;
const normalizeEvent = rawEvent => {
  (0, _invariant.invariant)(rawEvent, 'Event not found');
  const timestamp = normalizeEventTimestamp(rawEvent);
  const sequence = normalizeEventSequence(rawEvent);
  const level = normalizeLogLevel(rawEvent);
  const type = normalizeEventType(rawEvent);
  const executionId = normalizeExecutionId(rawEvent);
  const message = normalizeEventMessage(rawEvent, type);
  return {
    timestamp,
    sequence,
    level,
    type,
    message,
    execution_id: executionId
  };
};
const normalizeEventTimestamp = event => {
  (0, _invariant.invariant)(event['@timestamp'], 'Required "@timestamp" field is not found');
  return event['@timestamp'];
};
const normalizeEventSequence = event => {
  var _event$event;
  const value = (_event$event = event.event) === null || _event$event === void 0 ? void 0 : _event$event.sequence;
  if (typeof value === 'number') {
    return value;
  }
  if (typeof value === 'string') {
    return Number(value);
  }
  return 0;
};
const normalizeLogLevel = event => {
  var _event$log;
  const value = (_event$log = event.log) === null || _event$log === void 0 ? void 0 : _event$log.level;
  if (!value) {
    return _rule_monitoring.LogLevelEnum.debug;
  }
  const result = _rule_monitoring.LogLevel.safeParse(value);
  return result.success ? result.data : _rule_monitoring.LogLevelEnum.trace;
};
const normalizeEventType = event => {
  var _event$event2;
  const value = (_event$event2 = event.event) === null || _event$event2 === void 0 ? void 0 : _event$event2.action;
  (0, _invariant.invariant)(value, 'Required "event.action" field is not found');
  const result = _rule_monitoring.RuleExecutionEventType.safeParse(value);
  return result.success ? result.data : _rule_monitoring.RuleExecutionEventTypeEnum.message;
};
const normalizeEventMessage = (event, type) => {
  if (type === _rule_monitoring.RuleExecutionEventTypeEnum.message) {
    return event.message || '';
  }
  if (type === _rule_monitoring.RuleExecutionEventTypeEnum['status-change']) {
    var _event$kibana, _event$kibana$alert, _event$kibana$alert$r, _event$kibana$alert$r2, _event$kibana2, _event$kibana2$alert, _event$kibana2$alert$, _event$kibana2$alert$2;
    (0, _invariant.invariant)((_event$kibana = event.kibana) === null || _event$kibana === void 0 ? void 0 : (_event$kibana$alert = _event$kibana.alert) === null || _event$kibana$alert === void 0 ? void 0 : (_event$kibana$alert$r = _event$kibana$alert.rule) === null || _event$kibana$alert$r === void 0 ? void 0 : (_event$kibana$alert$r2 = _event$kibana$alert$r.execution) === null || _event$kibana$alert$r2 === void 0 ? void 0 : _event$kibana$alert$r2.status, 'Required "kibana.alert.rule.execution.status" field is not found');
    const status = (_event$kibana2 = event.kibana) === null || _event$kibana2 === void 0 ? void 0 : (_event$kibana2$alert = _event$kibana2.alert) === null || _event$kibana2$alert === void 0 ? void 0 : (_event$kibana2$alert$ = _event$kibana2$alert.rule) === null || _event$kibana2$alert$ === void 0 ? void 0 : (_event$kibana2$alert$2 = _event$kibana2$alert$.execution) === null || _event$kibana2$alert$2 === void 0 ? void 0 : _event$kibana2$alert$2.status;
    const message = event.message || '';
    return `Rule changed status to "${status}". ${message}`;
  }
  if (type === _rule_monitoring.RuleExecutionEventTypeEnum['execution-metrics']) {
    var _event$kibana3, _event$kibana3$alert, _event$kibana3$alert$, _event$kibana3$alert$2;
    (0, _invariant.invariant)((_event$kibana3 = event.kibana) === null || _event$kibana3 === void 0 ? void 0 : (_event$kibana3$alert = _event$kibana3.alert) === null || _event$kibana3$alert === void 0 ? void 0 : (_event$kibana3$alert$ = _event$kibana3$alert.rule) === null || _event$kibana3$alert$ === void 0 ? void 0 : (_event$kibana3$alert$2 = _event$kibana3$alert$.execution) === null || _event$kibana3$alert$2 === void 0 ? void 0 : _event$kibana3$alert$2.metrics, 'Required "kibana.alert.rule.execution.metrics" field is not found');
    return JSON.stringify(event.kibana.alert.rule.execution.metrics);
  }
  (0, _utility_types.assertUnreachable)(type);
  return '';
};
const normalizeExecutionId = event => {
  var _event$kibana4, _event$kibana4$alert, _event$kibana4$alert$, _event$kibana4$alert$2;
  (0, _invariant.invariant)((_event$kibana4 = event.kibana) === null || _event$kibana4 === void 0 ? void 0 : (_event$kibana4$alert = _event$kibana4.alert) === null || _event$kibana4$alert === void 0 ? void 0 : (_event$kibana4$alert$ = _event$kibana4$alert.rule) === null || _event$kibana4$alert$ === void 0 ? void 0 : (_event$kibana4$alert$2 = _event$kibana4$alert$.execution) === null || _event$kibana4$alert$2 === void 0 ? void 0 : _event$kibana4$alert$2.uuid, 'Required "kibana.alert.rule.execution.uuid" field is not found');
  return event.kibana.alert.rule.execution.uuid;
};
const buildEventLogKqlFilter = ({
  searchTerm,
  eventTypes,
  logLevels,
  dateStart,
  dateEnd
}) => {
  const filters = [`${f.EVENT_PROVIDER}:${_event_log_constants.RULE_EXECUTION_LOG_PROVIDER}`];
  if (searchTerm !== null && searchTerm !== void 0 && searchTerm.length) {
    filters.push(`${f.MESSAGE}:${(0, _kql.prepareKQLStringParam)(searchTerm)}`);
  }
  if (eventTypes !== null && eventTypes !== void 0 && eventTypes.length) {
    filters.push(`${f.EVENT_ACTION}:(${(0, _kql2.kqlOr)(eventTypes)})`);
  }
  if (logLevels !== null && logLevels !== void 0 && logLevels.length) {
    filters.push(`${f.LOG_LEVEL}:(${(0, _kql2.kqlOr)(logLevels)})`);
  }
  const dateRangeFilter = [];
  if (dateStart) {
    dateRangeFilter.push(`${f.TIMESTAMP} >= ${(0, _kql.prepareKQLStringParam)(dateStart)}`);
  }
  if (dateEnd) {
    dateRangeFilter.push(`${f.TIMESTAMP} <= ${(0, _kql.prepareKQLStringParam)(dateEnd)}`);
  }
  if (dateRangeFilter.length) {
    filters.push((0, _kql2.kqlAnd)(dateRangeFilter));
  }
  return (0, _kql2.kqlAnd)(filters);
};
const findRuleExecutionIds = async ({
  eventLog,
  soType,
  soIds,
  start,
  end,
  filter
}) => {
  var _value, _runTypesResponse$agg, _runTypesResponse$agg2, _runTypesResponse$agg3, _filteredExecutionUUI, _filteredExecutionUUI2;
  const runTypesResponse = await eventLog.aggregateEventsBySavedObjectIds(soType, soIds, {
    start,
    end,
    filter,
    aggs: {
      totalExecutions: {
        cardinality: {
          field: f.RULE_EXECUTION_UUID
        }
      },
      filteredExecutionUUIDs: {
        terms: {
          field: f.RULE_EXECUTION_UUID,
          order: {
            executeStartTime: 'desc'
          },
          size: _securitysolutionRules.MAX_EXECUTION_EVENTS_DISPLAYED
        },
        aggs: {
          executeStartTime: {
            min: {
              field: f.TIMESTAMP
            }
          }
        }
      }
    }
  });
  const total = (_value = (_runTypesResponse$agg = runTypesResponse.aggregations) === null || _runTypesResponse$agg === void 0 ? void 0 : (_runTypesResponse$agg2 = _runTypesResponse$agg.totalExecutions) === null || _runTypesResponse$agg2 === void 0 ? void 0 : _runTypesResponse$agg2.value) !== null && _value !== void 0 ? _value : 0;
  const filteredExecutionUUIDs = (_runTypesResponse$agg3 = runTypesResponse.aggregations) === null || _runTypesResponse$agg3 === void 0 ? void 0 : _runTypesResponse$agg3.filteredExecutionUUIDs;
  const executionIds = (_filteredExecutionUUI = filteredExecutionUUIDs === null || filteredExecutionUUIDs === void 0 ? void 0 : (_filteredExecutionUUI2 = filteredExecutionUUIDs.buckets) === null || _filteredExecutionUUI2 === void 0 ? void 0 : _filteredExecutionUUI2.map(b => b.key)) !== null && _filteredExecutionUUI !== void 0 ? _filteredExecutionUUI : [];
  return {
    executionIds,
    total
  };
};