"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.convertRuleResponseToAlertingRule = void 0;
var _utils = require("../../../../../../../common/detection_engine/rule_management/utils");
var _transform_actions = require("../../../../../../../common/detection_engine/transform_actions");
var _utils2 = require("../../../../../../../common/detection_engine/utils");
var _utility_types = require("../../../../../../../common/utility_types");
var _object_case_converters = require("../../../../../../utils/object_case_converters");
var _rule_actions = require("../../../normalization/rule_actions");
var _utils3 = require("../../../utils/utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * These are the fields that are added to the rule response that are not part of the rule params
 */

const convertRuleResponseToAlertingRule = (rule, actionsClient) => {
  var _ruleActions$map, _rule$response_action, _rule$output_index;
  const [ruleSystemActions, ruleActions] = (0, _utils3.separateActionsAndSystemAction)(actionsClient, rule.actions);
  const systemActions = ruleSystemActions === null || ruleSystemActions === void 0 ? void 0 : ruleSystemActions.map(action => (0, _transform_actions.transformRuleToAlertAction)(action));
  const alertActions = (_ruleActions$map = ruleActions === null || ruleActions === void 0 ? void 0 : ruleActions.map(action => (0, _transform_actions.transformRuleToAlertAction)(action))) !== null && _ruleActions$map !== void 0 ? _ruleActions$map : [];
  const actions = (0, _rule_actions.transformToActionFrequency)(alertActions, rule.throttle);
  const responseActions = (_rule$response_action = rule.response_actions) === null || _rule$response_action === void 0 ? void 0 : _rule$response_action.map(ruleResponseAction => (0, _transform_actions.transformRuleToAlertResponseAction)(ruleResponseAction));
  // Because of Omit<RuleResponse, RuntimeFields> Typescript doesn't recognize
  // that rule is assignable to TypeSpecificCreateProps despite omitted fields
  // are not part of type specific props. So we need to cast here.
  const typeSpecificParams = typeSpecificSnakeToCamel(rule);
  return {
    name: rule.name,
    tags: rule.tags,
    params: {
      author: rule.author,
      buildingBlockType: rule.building_block_type,
      description: rule.description,
      ruleId: rule.rule_id,
      falsePositives: rule.false_positives,
      from: rule.from,
      investigationFields: rule.investigation_fields,
      immutable: rule.immutable,
      ruleSource: rule.rule_source ? (0, _object_case_converters.convertObjectKeysToCamelCase)(rule.rule_source) : undefined,
      license: rule.license,
      outputIndex: (_rule$output_index = rule.output_index) !== null && _rule$output_index !== void 0 ? _rule$output_index : '',
      timelineId: rule.timeline_id,
      timelineTitle: rule.timeline_title,
      meta: rule.meta,
      maxSignals: rule.max_signals,
      relatedIntegrations: rule.related_integrations,
      requiredFields: (0, _utils.addEcsToRequiredFields)(rule.required_fields),
      riskScore: rule.risk_score,
      riskScoreMapping: rule.risk_score_mapping,
      ruleNameOverride: rule.rule_name_override,
      setup: rule.setup,
      severity: rule.severity,
      severityMapping: rule.severity_mapping,
      threat: rule.threat,
      timestampOverride: rule.timestamp_override,
      timestampOverrideFallbackDisabled: rule.timestamp_override_fallback_disabled,
      to: rule.to,
      references: rule.references,
      namespace: rule.namespace,
      note: rule.note,
      version: rule.version,
      exceptionsList: rule.exceptions_list,
      responseActions,
      ...typeSpecificParams
    },
    schedule: {
      interval: rule.interval
    },
    actions,
    ...(systemActions && {
      systemActions
    })
  };
};

// Converts params from the snake case API format to the internal camel case format AND applies default values where needed.
// Notice that params.language is possibly undefined for most rule types in the API but we default it to kuery to match
// the legacy API behavior
exports.convertRuleResponseToAlertingRule = convertRuleResponseToAlertingRule;
const typeSpecificSnakeToCamel = params => {
  switch (params.type) {
    case 'eql':
      {
        return {
          type: params.type,
          language: params.language,
          index: params.index,
          dataViewId: params.data_view_id,
          query: params.query,
          filters: params.filters,
          timestampField: params.timestamp_field,
          eventCategoryOverride: params.event_category_override,
          tiebreakerField: params.tiebreaker_field,
          alertSuppression: params.alert_suppression ? (0, _object_case_converters.convertObjectKeysToCamelCase)(params.alert_suppression) : undefined
        };
      }
    case 'esql':
      {
        return {
          type: params.type,
          language: params.language,
          query: params.query,
          alertSuppression: params.alert_suppression ? (0, _object_case_converters.convertObjectKeysToCamelCase)(params.alert_suppression) : undefined
        };
      }
    case 'threat_match':
      {
        var _params$language;
        return {
          type: params.type,
          language: (_params$language = params.language) !== null && _params$language !== void 0 ? _params$language : 'kuery',
          index: params.index,
          dataViewId: params.data_view_id,
          query: params.query,
          filters: params.filters,
          savedId: params.saved_id,
          threatFilters: params.threat_filters,
          threatQuery: params.threat_query,
          threatMapping: params.threat_mapping,
          threatLanguage: params.threat_language,
          threatIndex: params.threat_index,
          threatIndicatorPath: params.threat_indicator_path,
          concurrentSearches: params.concurrent_searches,
          itemsPerSearch: params.items_per_search,
          alertSuppression: params.alert_suppression ? (0, _object_case_converters.convertObjectKeysToCamelCase)(params.alert_suppression) : undefined
        };
      }
    case 'query':
      {
        var _params$language2, _params$query;
        return {
          type: params.type,
          language: (_params$language2 = params.language) !== null && _params$language2 !== void 0 ? _params$language2 : 'kuery',
          index: params.index,
          dataViewId: params.data_view_id,
          query: (_params$query = params.query) !== null && _params$query !== void 0 ? _params$query : '',
          filters: params.filters,
          savedId: params.saved_id,
          alertSuppression: params.alert_suppression ? (0, _object_case_converters.convertObjectKeysToCamelCase)(params.alert_suppression) : undefined
        };
      }
    case 'saved_query':
      {
        var _params$language3;
        return {
          type: params.type,
          language: (_params$language3 = params.language) !== null && _params$language3 !== void 0 ? _params$language3 : 'kuery',
          index: params.index,
          query: params.query,
          filters: params.filters,
          savedId: params.saved_id,
          dataViewId: params.data_view_id,
          alertSuppression: params.alert_suppression ? (0, _object_case_converters.convertObjectKeysToCamelCase)(params.alert_suppression) : undefined
        };
      }
    case 'threshold':
      {
        var _params$language4, _params$alert_suppres;
        return {
          type: params.type,
          language: (_params$language4 = params.language) !== null && _params$language4 !== void 0 ? _params$language4 : 'kuery',
          index: params.index,
          dataViewId: params.data_view_id,
          query: params.query,
          filters: params.filters,
          savedId: params.saved_id,
          threshold: (0, _utils2.normalizeThresholdObject)(params.threshold),
          alertSuppression: (_params$alert_suppres = params.alert_suppression) !== null && _params$alert_suppres !== void 0 && _params$alert_suppres.duration ? {
            duration: params.alert_suppression.duration
          } : undefined
        };
      }
    case 'machine_learning':
      {
        return {
          type: params.type,
          anomalyThreshold: params.anomaly_threshold,
          machineLearningJobId: (0, _utils2.normalizeMachineLearningJobIds)(params.machine_learning_job_id),
          alertSuppression: params.alert_suppression ? (0, _object_case_converters.convertObjectKeysToCamelCase)(params.alert_suppression) : undefined
        };
      }
    case 'new_terms':
      {
        var _params$language5;
        return {
          type: params.type,
          query: params.query,
          newTermsFields: params.new_terms_fields,
          historyWindowStart: params.history_window_start,
          index: params.index,
          filters: params.filters,
          language: (_params$language5 = params.language) !== null && _params$language5 !== void 0 ? _params$language5 : 'kuery',
          dataViewId: params.data_view_id,
          alertSuppression: params.alert_suppression ? (0, _object_case_converters.convertObjectKeysToCamelCase)(params.alert_suppression) : undefined
        };
      }
    default:
      {
        return (0, _utility_types.assertUnreachable)(params);
      }
  }
};