"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.withEndpointAuthz = void 0;
var _error_handler = require("./error_handler");
var _stringify = require("../utils/stringify");
var _errors = require("../errors");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Interface for defining the needed permissions to access an API. Both sets of permissions (if defined) will
 * be `AND` together.
 */

/**
 * Wraps an API route handler and handles authorization checks for endpoint related
 * apis.
 * @param neededAuthz
 * @param routeHandler
 * @param logger
 * @param additionalChecks
 */
const withEndpointAuthz = (neededAuthz, logger, routeHandler, additionalChecks) => {
  var _neededAuthz$all, _neededAuthz$any;
  const needAll = (_neededAuthz$all = neededAuthz.all) !== null && _neededAuthz$all !== void 0 ? _neededAuthz$all : [];
  const needAny = (_neededAuthz$any = neededAuthz.any) !== null && _neededAuthz$any !== void 0 ? _neededAuthz$any : [];
  const validateAll = needAll.length > 0;
  const validateAny = needAny.length > 0;
  const enforceAuthz = validateAll || validateAny;
  const logAuthzFailure = (user, authzValidationResults, needed) => {
    logger.debug(`Unauthorized: user ${user} ${needed === needAll ? 'needs ALL' : 'needs at least one'} of the following privileges:\n${(0, _stringify.stringify)(needed)}\nbut is missing: ${(0, _stringify.stringify)(Object.entries(authzValidationResults).filter(([_, value]) => !value).map(([key]) => key))}`);
  };
  if (!enforceAuthz) {
    var _Error$stack;
    logger.warn(`Authorization disabled for API route: ${(_Error$stack = new Error('').stack) !== null && _Error$stack !== void 0 ? _Error$stack : '?'}`);
  }
  const handlerWrapper = async (context, request, response) => {
    if (enforceAuthz) {
      const coreServices = await context.core;
      const endpointAuthz = await (await context.securitySolution).getEndpointAuthz();
      let authzValidationResults = {};
      const permissionChecker = permission => {
        authzValidationResults[permission] = endpointAuthz[permission];
        return endpointAuthz[permission];
      };

      // has `all`?
      if (validateAll && !needAll.every(permissionChecker)) {
        var _coreServices$securit, _coreServices$securit2;
        logAuthzFailure((_coreServices$securit = (_coreServices$securit2 = coreServices.security.authc.getCurrentUser()) === null || _coreServices$securit2 === void 0 ? void 0 : _coreServices$securit2.username) !== null && _coreServices$securit !== void 0 ? _coreServices$securit : '', authzValidationResults, needAll);
        return response.forbidden({
          body: new _errors.EndpointAuthorizationError({
            need_all: [...needAll]
          })
        });
      }
      authzValidationResults = {};

      // has `any`?
      if (validateAny && !needAny.some(permissionChecker)) {
        var _coreServices$securit3, _coreServices$securit4;
        logAuthzFailure((_coreServices$securit3 = (_coreServices$securit4 = coreServices.security.authc.getCurrentUser()) === null || _coreServices$securit4 === void 0 ? void 0 : _coreServices$securit4.username) !== null && _coreServices$securit3 !== void 0 ? _coreServices$securit3 : '', authzValidationResults, needAny);
        return response.forbidden({
          body: new _errors.EndpointAuthorizationError({
            need_any: [...needAny]
          })
        });
      }
    }
    if (additionalChecks) {
      try {
        await additionalChecks(context, request);
      } catch (err) {
        logger.debug(() => (0, _stringify.stringify)(err));
        return (0, _error_handler.errorHandler)(logger, response, err);
      }
    }

    // Authz is good call the route handler
    return routeHandler(context, request, response);
  };
  return handlerWrapper;
};
exports.withEndpointAuthz = withEndpointAuthz;