"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.INTEGRATION_KNOWLEDGE_TOOL = void 0;
var _tools = require("@langchain/core/tools");
var _zod = require("@kbn/zod");
var _elasticAssistantCommon = require("@kbn/elastic-assistant-common");
var _references = require("@kbn/elastic-assistant-common/impl/content_references/references");
var _document = require("langchain/document");
var _common = require("../../../../common");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const INTEGRATIONS_BASE_PATH = '/app/integrations/detail';
const toolDetails = {
  // note: this description is overwritten when `getTool` is called
  // local definitions exist ../elastic_assistant/server/lib/prompt/tool_prompts.ts
  // local definitions can be overwritten by security-ai-prompt integration definitions
  description: 'Call this for knowledge from Fleet-installed integrations, which contains information on how to configure and use integrations for data ingestion.',
  id: 'integration-knowledge-tool',
  name: 'IntegrationKnowledgeTool'
};
const INTEGRATION_KNOWLEDGE_INDEX = '.integration_knowledge';
const INTEGRATION_KNOWLEDGE_TOOL = exports.INTEGRATION_KNOWLEDGE_TOOL = {
  ...toolDetails,
  sourceRegister: _common.APP_UI_ID,
  isSupported: params => {
    const {
      assistantContext
    } = params;
    return assistantContext != null;
  },
  async getTool(params) {
    if (!this.isSupported(params)) return null;
    const {
      assistantContext,
      contentReferencesStore
    } = params;

    // Check if the .integration_knowledge index exists before registering the tool
    // This has to be done with `.search` since `.exists` and `.get` can't be performed
    // with the internal system user (lack of permissions)
    try {
      const indexExists = await assistantContext.core.elasticsearch.client.asInternalUser.search({
        index: INTEGRATION_KNOWLEDGE_INDEX,
        size: 0
      });
      if (!indexExists) {
        return null;
      }
    } catch (error) {
      // If there's an error checking the index, assume it doesn't exist and don't register the tool
      return null;
    }
    return (0, _tools.tool)(async input => {
      try {
        // Search the .integration_knowledge index using semantic search on the content field
        const response = await assistantContext.core.elasticsearch.client.asInternalUser.search({
          index: INTEGRATION_KNOWLEDGE_INDEX,
          size: 10,
          query: {
            semantic: {
              field: 'content',
              query: input.question
            }
          },
          _source: ['package_name', 'filename', 'content', 'version']
        });
        const citedDocs = response.hits.hits.map(hit => {
          const source = hit._source;
          let reference;
          try {
            // Create a reference to the integration details page
            const packageUrl = `${assistantContext.getServerBasePath()}${INTEGRATIONS_BASE_PATH}/${source.package_name}`;
            const title = `${source.package_name} integration (${source.filename})`;
            reference = contentReferencesStore.add(p => (0, _references.hrefReference)(p.id, packageUrl, title));
          } catch (_error) {
            reference = contentReferencesStore.add(p => (0, _references.knowledgeBaseReference)(p.id, `Integration knowledge for ${source.package_name}`, 'integrationKnowledge'));
          }
          return new _document.Document({
            id: hit._id,
            pageContent: `${(0, _elasticAssistantCommon.contentReferenceString)(reference)}\n\nPackage: ${source.package_name}${source.version ? ` (v${source.version})` : ''}\nFile: ${source.filename}\n${source.content}`,
            metadata: {
              package_name: source.package_name,
              package_version: source.version,
              filename: source.filename
            }
          });
        });

        // TODO: Token pruning
        const result = JSON.stringify(citedDocs).substring(0, 20000);
        return result;
      } catch (error) {
        return `Error querying integration knowledge: ${error.message}. The integration knowledge base may not be available.`;
      }
    }, {
      name: toolDetails.name,
      description: params.description || toolDetails.description,
      schema: _zod.z.object({
        question: _zod.z.string().describe('Key terms to retrieve Fleet-installed integration knowledge for, like specific integration names, configuration questions, or data ingestion topics.')
      }),
      tags: ['integration', 'knowledge-base', 'fleet']
    });
  }
};