"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.usePrevalence = void 0;
var _react = require("react");
var _use_highlighted_fields = require("./use_highlighted_fields");
var _highlighted_fields_helpers = require("../utils/highlighted_fields_helpers");
var _use_fetch_prevalence = require("./use_fetch_prevalence");
var _event_kinds = require("../constants/event_kinds");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Hook to fetch the prevalence data, then prepares the data to be consumed by the EuiInMemoryTable component
 * in the PrevalenceDetails component
 */
const usePrevalence = ({
  interval,
  dataFormattedForFieldBrowser,
  investigationFields
}) => {
  const highlightedFields = (0, _use_highlighted_fields.useHighlightedFields)({
    dataFormattedForFieldBrowser,
    investigationFields
  });
  const highlightedFieldsFilters = (0, _react.useMemo)(() => (0, _highlighted_fields_helpers.convertHighlightedFieldsToPrevalenceFilters)(highlightedFields), [highlightedFields]);
  const {
    data,
    loading,
    error
  } = (0, _use_fetch_prevalence.useFetchPrevalence)({
    highlightedFieldsFilters,
    interval
  });
  const items = [];
  if (data && data.aggregations) {
    // total number of unique hosts in the environment
    const uniqueHostsInEnvironment = data.aggregations[_use_fetch_prevalence.HOSTS_AGG_KEY].value;

    // total number of unique users in the environment
    const uniqueUsersInEnvironment = data.aggregations[_use_fetch_prevalence.USERS_AGG_KEY].value;
    const fieldNames = Object.keys(data.aggregations[_use_fetch_prevalence.FIELD_NAMES_AGG_KEY].buckets);
    fieldNames.forEach(fieldName => {
      var _data$aggregations$FI;
      const fieldValues = highlightedFields[fieldName].values;

      // retrieves the number of signals for the current field/value pair
      const alertCount = ((_data$aggregations$FI = data.aggregations[_use_fetch_prevalence.FIELD_NAMES_AGG_KEY].buckets[fieldName][_use_fetch_prevalence.EVENT_KIND_AGG_KEY].buckets.find(aggregationValue => aggregationValue.key === _event_kinds.EventKind.signal)) === null || _data$aggregations$FI === void 0 ? void 0 : _data$aggregations$FI.doc_count) || 0;

      // calculate the number of documents (non-signal) for the current field/value pair
      let docCount = 0;
      data.aggregations[_use_fetch_prevalence.FIELD_NAMES_AGG_KEY].buckets[fieldName][_use_fetch_prevalence.EVENT_KIND_AGG_KEY].buckets.reduce((acc, curr) => {
        if (curr.key !== _event_kinds.EventKind.signal) {
          docCount += curr.doc_count;
        }
        return acc;
      }, docCount);

      // number of unique hosts in which the current field/value pair is present
      const uniqueHostsForCurrentFieldValuePair = data.aggregations[_use_fetch_prevalence.FIELD_NAMES_AGG_KEY].buckets[fieldName][_use_fetch_prevalence.HOST_NAME_AGG_KEY].value;

      // number of unique users in which the current field/value pair is present
      const uniqueUsersForCurrentFieldValuePair = data.aggregations[_use_fetch_prevalence.FIELD_NAMES_AGG_KEY].buckets[fieldName][_use_fetch_prevalence.USER_NAME_AGG_KEY].value;

      // calculate host prevalence
      const hostPrevalence = uniqueHostsInEnvironment ? uniqueHostsForCurrentFieldValuePair / uniqueHostsInEnvironment : 0;

      // calculate user prevalence
      const userPrevalence = uniqueUsersInEnvironment ? uniqueUsersForCurrentFieldValuePair / uniqueUsersInEnvironment : 0;
      items.push({
        field: fieldName,
        values: fieldValues,
        alertCount,
        docCount,
        hostPrevalence,
        userPrevalence
      });
    });
  }
  return {
    loading,
    error,
    data: items
  };
};
exports.usePrevalence = usePrevalence;