"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useCloseAlertsFromExceptions = void 0;
var _react = require("react");
var _default_config = require("../../../detections/components/alerts_table/default_config");
var _get_es_query_filter = require("../utils/get_es_query_filter");
var _helpers = require("../utils/helpers");
var i18nCommon = _interopRequireWildcard(require("../../../common/translations"));
var i18n = _interopRequireWildcard(require("./translations"));
var _use_app_toasts = require("../../../common/hooks/use_app_toasts");
var _update_alerts = require("../../../common/components/toolbar/bulk_actions/update_alerts");
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && {}.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Closes alerts.
 *
 * @param ruleStaticIds static id of the rules (rule.ruleId, not rule.id) where the exception updates will be applied
 * @param exceptionItems array of ExceptionListItemSchema to add or update
 * @param alertIdToClose - optional string representing alert to close
 * @param bulkCloseIndex - optional index used to create bulk close query
 *
 */

/**
 * Hook for closing alerts from exceptions
 */
const useCloseAlertsFromExceptions = () => {
  const {
    addSuccess,
    addError,
    addWarning
  } = (0, _use_app_toasts.useAppToasts)();
  const [isLoading, setIsLoading] = (0, _react.useState)(false);
  const closeAlertsRef = (0, _react.useRef)(null);
  (0, _react.useEffect)(() => {
    let isSubscribed = true;
    const abortCtrl = new AbortController();
    const onUpdateAlerts = async (ruleStaticIds, exceptionItems, alertIdToClose, bulkCloseIndex) => {
      try {
        var _alertIdResponse$upda, _alertIdResponse, _bulkResponse$updated, _bulkResponse, _alertIdResponse$vers, _alertIdResponse2, _bulkResponse$version, _bulkResponse2;
        setIsLoading(true);
        let alertIdResponse;
        let bulkResponse;
        if (alertIdToClose != null) {
          alertIdResponse = await (0, _update_alerts.updateAlertStatus)({
            signalIds: [alertIdToClose],
            status: 'closed',
            signal: abortCtrl.signal
          });
        }
        if (bulkCloseIndex != null) {
          const alertStatusFilter = (0, _default_config.buildAlertStatusesFilter)(['open', 'acknowledged', 'in-progress']);
          const filterByRuleIds = (0, _default_config.buildAlertsFilterByRuleIds)(ruleStaticIds);
          const filter = await (0, _get_es_query_filter.getEsQueryFilter)('', 'kuery', [...filterByRuleIds, ...alertStatusFilter], bulkCloseIndex, (0, _helpers.prepareExceptionItemsForBulkClose)(exceptionItems), false);
          bulkResponse = await (0, _update_alerts.updateAlertStatus)({
            query: filter,
            status: 'closed',
            signal: abortCtrl.signal
          });
        }

        // NOTE: there could be some overlap here... it's possible that the first response had conflicts
        // but that the alert was closed in the second call. In this case, a conflict will be reported even
        // though it was already resolved. I'm not sure that there's an easy way to solve this, but it should
        // have minimal impact on the user... they'd see a warning that indicates a possible conflict, but the
        // state of the alerts and their representation in the UI would be consistent.
        const updated = ((_alertIdResponse$upda = (_alertIdResponse = alertIdResponse) === null || _alertIdResponse === void 0 ? void 0 : _alertIdResponse.updated) !== null && _alertIdResponse$upda !== void 0 ? _alertIdResponse$upda : 0) + ((_bulkResponse$updated = (_bulkResponse = bulkResponse) === null || _bulkResponse === void 0 ? void 0 : _bulkResponse.updated) !== null && _bulkResponse$updated !== void 0 ? _bulkResponse$updated : 0);
        const conflicts = (_alertIdResponse$vers = (_alertIdResponse2 = alertIdResponse) === null || _alertIdResponse2 === void 0 ? void 0 : _alertIdResponse2.version_conflicts) !== null && _alertIdResponse$vers !== void 0 ? _alertIdResponse$vers : 0 + ((_bulkResponse$version = (_bulkResponse2 = bulkResponse) === null || _bulkResponse2 === void 0 ? void 0 : _bulkResponse2.version_conflicts) !== null && _bulkResponse$version !== void 0 ? _bulkResponse$version : 0);
        if (isSubscribed) {
          setIsLoading(false);
          addSuccess(i18n.CLOSE_ALERTS_SUCCESS(updated));
          if (conflicts > 0) {
            addWarning({
              title: i18nCommon.UPDATE_ALERT_STATUS_FAILED(conflicts),
              text: i18nCommon.UPDATE_ALERT_STATUS_FAILED_DETAILED(updated, conflicts)
            });
          }
        }
      } catch (error) {
        if (isSubscribed) {
          setIsLoading(false);
          addError(error, {
            title: i18n.CLOSE_ALERTS_ERROR
          });
        }
      }
    };
    closeAlertsRef.current = onUpdateAlerts;
    return () => {
      isSubscribed = false;
      abortCtrl.abort();
    };
  }, [addSuccess, addError, addWarning]);
  return [isLoading, closeAlertsRef.current];
};
exports.useCloseAlertsFromExceptions = useCloseAlertsFromExceptions;