"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useMissingPrivileges = void 0;
var _react = require("react");
var _constants = require("../../../common/constants");
var _user_info = require("../../detections/components/user_info");
var _user_privileges = require("../components/user_privileges");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const REQUIRED_INDEX_PRIVILEGES = ['read', 'write', 'view_index_metadata', 'manage'];
const getIndexName = indexPrivileges => {
  const [indexName] = Object.keys(indexPrivileges);
  return indexName;
};
const getMissingIndexPrivileges = indexPrivileges => {
  const indexName = getIndexName(indexPrivileges);
  const privileges = indexPrivileges[indexName];
  const missingPrivileges = REQUIRED_INDEX_PRIVILEGES.filter(privilege => !privileges[privilege]);
  if (missingPrivileges.length) {
    return [indexName, missingPrivileges];
  }
};
/**
 * Hook that returns index and feature privileges that are missing for the user.
 */
const useMissingPrivileges = () => {
  const {
    detectionEnginePrivileges,
    listPrivileges
  } = (0, _user_privileges.useUserPrivileges)();
  const [{
    canUserCRUD
  }] = (0, _user_info.useUserData)();
  return (0, _react.useMemo)(() => {
    const featurePrivileges = [];
    const indexPrivileges = [];
    if (canUserCRUD == null || listPrivileges.result == null || detectionEnginePrivileges.result == null) {
      /**
       * Do not check privileges till we get all the data. That helps to reduce
       * subsequent layout shift while loading and skip unneeded re-renders.
       */
      return {
        featurePrivileges,
        indexPrivileges
      };
    }
    if (!canUserCRUD) {
      featurePrivileges.push([_constants.SECURITY_FEATURE_ID, ['all']]);
    }
    const missingItemsPrivileges = getMissingIndexPrivileges(listPrivileges.result.listItems.index);
    if (missingItemsPrivileges) {
      indexPrivileges.push(missingItemsPrivileges);
    }
    const missingListsPrivileges = getMissingIndexPrivileges(listPrivileges.result.lists.index);
    if (missingListsPrivileges) {
      indexPrivileges.push(missingListsPrivileges);
    }
    const missingDetectionPrivileges = getMissingIndexPrivileges(detectionEnginePrivileges.result.index);
    if (missingDetectionPrivileges) {
      indexPrivileges.push(missingDetectionPrivileges);
    }
    return {
      featurePrivileges,
      indexPrivileges
    };
  }, [canUserCRUD, listPrivileges, detectionEnginePrivileges]);
};
exports.useMissingPrivileges = useMissingPrivileges;