"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.TagsClient = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _ebtTools = require("@kbn/ebt-tools");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const BULK_DELETE_TAG_EVENT = 'bulkDeleteTag';
const CREATE_TAG_EVENT = 'createTag';
const DELETE_TAG_EVENT = 'deleteTag';
const GET_ALL_TAGS_EVENT = 'getAllTag';
const FIND_TAG_EVENT = 'findTag';
const UPDATE_TAG_EVENT = 'updateTag';
const trapErrors = fn => {
  try {
    fn();
  } catch (e) {
    // trap
  }
};
class TagsClient {
  constructor({
    analytics,
    http,
    changeListener
  }) {
    (0, _defineProperty2.default)(this, "analytics", void 0);
    (0, _defineProperty2.default)(this, "http", void 0);
    (0, _defineProperty2.default)(this, "changeListener", void 0);
    this.analytics = analytics;
    this.http = http;
    this.changeListener = changeListener;
  }

  // public APIs from ITagsClient

  async create(attributes) {
    const startTime = window.performance.now();
    const {
      tag
    } = await this.http.post('/api/saved_objects_tagging/tags/create', {
      body: JSON.stringify(attributes)
    });
    const duration = window.performance.now() - startTime;
    (0, _ebtTools.reportPerformanceMetricEvent)(this.analytics, {
      eventName: CREATE_TAG_EVENT,
      duration
    });
    trapErrors(() => {
      if (this.changeListener) {
        this.changeListener.onCreate(tag);
      }
    });
    return tag;
  }
  async update(id, attributes) {
    const startTime = window.performance.now();
    const {
      tag
    } = await this.http.post(`/api/saved_objects_tagging/tags/${id}`, {
      body: JSON.stringify(attributes)
    });
    const duration = window.performance.now() - startTime;
    (0, _ebtTools.reportPerformanceMetricEvent)(this.analytics, {
      eventName: UPDATE_TAG_EVENT,
      duration
    });
    trapErrors(() => {
      if (this.changeListener) {
        const {
          id: newId,
          ...newAttributes
        } = tag;
        this.changeListener.onUpdate(newId, newAttributes);
      }
    });
    return tag;
  }
  async get(id) {
    const {
      tag
    } = await this.http.get(`/api/saved_objects_tagging/tags/${id}`);
    return tag;
  }
  async getAll({
    asSystemRequest
  } = {}) {
    const startTime = window.performance.now();
    const fetchOptions = {
      asSystemRequest
    };
    const {
      tags
    } = await this.http.get('/api/saved_objects_tagging/tags', fetchOptions);
    const duration = window.performance.now() - startTime;
    (0, _ebtTools.reportPerformanceMetricEvent)(this.analytics, {
      eventName: GET_ALL_TAGS_EVENT,
      duration
    });
    trapErrors(() => {
      if (this.changeListener) {
        this.changeListener.onGetAll(tags);
      }
    });
    return tags;
  }
  async delete(id) {
    const startTime = window.performance.now();
    await this.http.delete(`/api/saved_objects_tagging/tags/${id}`);
    const duration = window.performance.now() - startTime;
    (0, _ebtTools.reportPerformanceMetricEvent)(this.analytics, {
      eventName: DELETE_TAG_EVENT,
      duration
    });
    trapErrors(() => {
      if (this.changeListener) {
        this.changeListener.onDelete(id);
      }
    });
  }

  // internal APIs from ITagInternalClient

  async find({
    page,
    perPage,
    search
  }) {
    const startTime = window.performance.now();
    const response = await this.http.get('/internal/saved_objects_tagging/tags/_find', {
      query: {
        page,
        perPage,
        search
      }
    });
    const duration = window.performance.now() - startTime;
    (0, _ebtTools.reportPerformanceMetricEvent)(this.analytics, {
      eventName: FIND_TAG_EVENT,
      duration
    });
    return response;
  }
  async findByName(name, {
    exact
  } = {
    exact: false
  }) {
    const {
      tags = []
    } = await this.find({
      page: 1,
      perPage: 10000,
      search: name
    });
    if (exact) {
      const tag = tags.find(t => t.name.toLocaleLowerCase() === name.toLocaleLowerCase());
      return tag !== null && tag !== void 0 ? tag : null;
    }
    return tags.length > 0 ? tags[0] : null;
  }
  async bulkDelete(tagIds) {
    const startTime = window.performance.now();
    await this.http.post('/internal/saved_objects_tagging/tags/_bulk_delete', {
      body: JSON.stringify({
        ids: tagIds
      })
    });
    const duration = window.performance.now() - startTime;
    (0, _ebtTools.reportPerformanceMetricEvent)(this.analytics, {
      eventName: BULK_DELETE_TAG_EVENT,
      duration
    });
    trapErrors(() => {
      if (this.changeListener) {
        tagIds.forEach(tagId => {
          this.changeListener.onDelete(tagId);
        });
      }
    });
  }
}
exports.TagsClient = TagsClient;