"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.AlertsClient = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _boom = _interopRequireDefault(require("@hapi/boom"));
var _uuid = require("uuid");
var _esQuery = require("@kbn/es-query");
var _securitysolutionEsUtils = require("@kbn/securitysolution-es-utils");
var _ruleDataUtils = require("@kbn/rule-data-utils");
var _server = require("@kbn/alerting-plugin/server");
var _server2 = require("@kbn/data-plugin/server");
var _lodash = require("lodash");
var _lib = require("@kbn/alerting-plugin/server/lib");
var _technical_rule_data_field_names = require("../../common/technical_rule_data_field_names");
var _lib2 = require("../lib");
var _browser_fields = require("./browser_fields");
var _constants = require("./constants");
var _get_rule_type_ids_filter = require("../lib/get_rule_type_ids_filter");
var _get_consumers_filter = require("../lib/get_consumers_filter");
var _unique_fields = require("../utils/unique_fields");
var _get_alert_fields_from_index_fetcher = require("../utils/get_alert_fields_from_index_fetcher");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

// TODO: Fix typings https://github.com/elastic/kibana/issues/101776

const isValidAlert = source => {
  var _source$_source, _source$_source2, _source$_source3, _source$fields, _source$fields2, _source$fields3;
  return (source === null || source === void 0 ? void 0 : (_source$_source = source._source) === null || _source$_source === void 0 ? void 0 : _source$_source[_technical_rule_data_field_names.ALERT_RULE_TYPE_ID]) != null && (source === null || source === void 0 ? void 0 : (_source$_source2 = source._source) === null || _source$_source2 === void 0 ? void 0 : _source$_source2[_technical_rule_data_field_names.ALERT_RULE_CONSUMER]) != null && (source === null || source === void 0 ? void 0 : (_source$_source3 = source._source) === null || _source$_source3 === void 0 ? void 0 : _source$_source3[_technical_rule_data_field_names.SPACE_IDS]) != null || (source === null || source === void 0 ? void 0 : (_source$fields = source.fields) === null || _source$fields === void 0 ? void 0 : _source$fields[_technical_rule_data_field_names.ALERT_RULE_TYPE_ID][0]) != null && (source === null || source === void 0 ? void 0 : (_source$fields2 = source.fields) === null || _source$fields2 === void 0 ? void 0 : _source$fields2[_technical_rule_data_field_names.ALERT_RULE_CONSUMER][0]) != null && (source === null || source === void 0 ? void 0 : (_source$fields3 = source.fields) === null || _source$fields3 === void 0 ? void 0 : _source$fields3[_technical_rule_data_field_names.SPACE_IDS][0]) != null;
};
/**
 * Provides apis to interact with alerts as data
 * ensures the request is authorized to perform read / write actions
 * on alerts as data.
 */
class AlertsClient {
  constructor(options) {
    (0, _defineProperty2.default)(this, "logger", void 0);
    (0, _defineProperty2.default)(this, "auditLogger", void 0);
    (0, _defineProperty2.default)(this, "authorization", void 0);
    (0, _defineProperty2.default)(this, "esClient", void 0);
    (0, _defineProperty2.default)(this, "esClientScoped", void 0);
    (0, _defineProperty2.default)(this, "spaceId", void 0);
    (0, _defineProperty2.default)(this, "ruleDataService", void 0);
    (0, _defineProperty2.default)(this, "getRuleList", void 0);
    (0, _defineProperty2.default)(this, "getAlertIndicesAlias", void 0);
    this.logger = options.logger;
    this.authorization = options.authorization;
    this.esClient = options.esClient;
    this.esClientScoped = options.esClientScoped;
    this.auditLogger = options.auditLogger;
    // If spaceId is undefined, it means that spaces is disabled
    // Otherwise, if space is enabled and not specified, it is "default"
    this.spaceId = this.authorization.getSpaceId();
    this.ruleDataService = options.ruleDataService;
    this.getRuleList = options.getRuleList;
    this.getAlertIndicesAlias = options.getAlertIndicesAlias;
  }
  getOutcome(operation) {
    return {
      outcome: operation === _server.WriteOperations.Update ? 'unknown' : 'success'
    };
  }
  getAlertStatusFieldUpdate(source, status) {
    return (source === null || source === void 0 ? void 0 : source[_technical_rule_data_field_names.ALERT_WORKFLOW_STATUS]) == null ? {
      signal: {
        status
      }
    } : {
      [_technical_rule_data_field_names.ALERT_WORKFLOW_STATUS]: status
    };
  }
  getAlertCaseIdsFieldUpdate(source, caseIds) {
    var _source$ALERT_CASE_ID;
    const uniqueCaseIds = new Set([...((_source$ALERT_CASE_ID = source === null || source === void 0 ? void 0 : source[_ruleDataUtils.ALERT_CASE_IDS]) !== null && _source$ALERT_CASE_ID !== void 0 ? _source$ALERT_CASE_ID : []), ...caseIds]);
    return {
      [_ruleDataUtils.ALERT_CASE_IDS]: Array.from(uniqueCaseIds.values())
    };
  }
  validateTotalCasesPerAlert(source, caseIds) {
    var _source$ALERT_CASE_ID2;
    const currentCaseIds = (_source$ALERT_CASE_ID2 = source === null || source === void 0 ? void 0 : source[_ruleDataUtils.ALERT_CASE_IDS]) !== null && _source$ALERT_CASE_ID2 !== void 0 ? _source$ALERT_CASE_ID2 : [];
    if (currentCaseIds.length + caseIds.length > _ruleDataUtils.MAX_CASES_PER_ALERT) {
      throw _boom.default.badRequest(`You cannot attach more than ${_ruleDataUtils.MAX_CASES_PER_ALERT} cases to an alert`);
    }
  }

  /**
   * Accepts an array of ES documents and executes ensureAuthorized for the given operation
   */
  async ensureAllAuthorized(items, operation) {
    const {
      hitIds,
      ownersAndRuleTypeIds
    } = items.reduce((acc, hit) => {
      var _hit$_source, _hit$_source2;
      return {
        hitIds: [hit._id, ...acc.hitIds],
        ownersAndRuleTypeIds: [{
          [_technical_rule_data_field_names.ALERT_RULE_TYPE_ID]: hit === null || hit === void 0 ? void 0 : (_hit$_source = hit._source) === null || _hit$_source === void 0 ? void 0 : _hit$_source[_technical_rule_data_field_names.ALERT_RULE_TYPE_ID],
          [_technical_rule_data_field_names.ALERT_RULE_CONSUMER]: hit === null || hit === void 0 ? void 0 : (_hit$_source2 = hit._source) === null || _hit$_source2 === void 0 ? void 0 : _hit$_source2[_technical_rule_data_field_names.ALERT_RULE_CONSUMER]
        }]
      };
    }, {
      hitIds: [],
      ownersAndRuleTypeIds: []
    });
    const assertString = hit => hit !== null && hit !== undefined;
    return Promise.all(ownersAndRuleTypeIds.map(hit => {
      const alertOwner = hit === null || hit === void 0 ? void 0 : hit[_technical_rule_data_field_names.ALERT_RULE_CONSUMER];
      const ruleId = hit === null || hit === void 0 ? void 0 : hit[_technical_rule_data_field_names.ALERT_RULE_TYPE_ID];
      if (hit != null && assertString(alertOwner) && assertString(ruleId)) {
        return this.authorization.ensureAuthorized({
          ruleTypeId: ruleId,
          consumer: alertOwner,
          operation,
          entity: _server.AlertingAuthorizationEntity.Alert
        });
      }
    })).catch(error => {
      for (const hitId of hitIds) {
        var _this$auditLogger;
        (_this$auditLogger = this.auditLogger) === null || _this$auditLogger === void 0 ? void 0 : _this$auditLogger.log((0, _lib.alertAuditEvent)({
          action: _lib.operationAlertAuditActionMap[operation],
          id: hitId,
          error
        }));
      }
      throw error;
    });
  }

  /**
   * Searches alerts by id or query and audits the results
   */
  async searchAlerts({
    id,
    query,
    aggs,
    _source,
    track_total_hits: trackTotalHits,
    size,
    index,
    operation,
    sort,
    lastSortIds = [],
    ruleTypeIds,
    consumers,
    runtimeMappings
  }) {
    try {
      var _result$hits, _result$hits$hits;
      const alertSpaceId = this.spaceId;
      if (alertSpaceId == null) {
        const errorMessage = 'Failed to acquire spaceId from authorization client';
        this.logger.error(`fetchAlertAndAudit threw an error: ${errorMessage}`);
        throw _boom.default.failedDependency(`fetchAlertAndAudit threw an error: ${errorMessage}`);
      }
      const config = (0, _ruleDataUtils.getEsQueryConfig)();
      let queryBody = {
        fields: [_technical_rule_data_field_names.ALERT_RULE_TYPE_ID, _technical_rule_data_field_names.ALERT_RULE_CONSUMER, _technical_rule_data_field_names.ALERT_WORKFLOW_STATUS, _technical_rule_data_field_names.SPACE_IDS],
        query: await this.buildEsQueryWithAuthz(query, id, alertSpaceId, operation, config, ruleTypeIds, consumers),
        aggs,
        _source,
        track_total_hits: trackTotalHits,
        size,
        sort: sort || [{
          '@timestamp': {
            order: 'asc',
            unmapped_type: 'date'
          }
        }],
        runtime_mappings: runtimeMappings
      };
      if (lastSortIds.length > 0) {
        queryBody = {
          ...queryBody,
          search_after: lastSortIds
        };
      }
      const result = await this.esClient.search({
        index: index !== null && index !== void 0 ? index : '.alerts-*',
        ignore_unavailable: true,
        ...queryBody,
        seq_no_primary_term: true
      });
      if (!(result !== null && result !== void 0 && (_result$hits = result.hits) !== null && _result$hits !== void 0 && (_result$hits$hits = _result$hits.hits) !== null && _result$hits$hits !== void 0 && _result$hits$hits.length)) {
        return result;
      }
      if (result.hits.hits.some(hit => !isValidAlert(hit))) {
        const errorMessage = `Invalid alert found with id of "${id}" or with query "${query}" and operation ${operation}`;
        this.logger.error(errorMessage);
        throw _boom.default.badData(errorMessage);
      }

      // @ts-expect-error type mismatch: SearchHit._id is optional
      await this.ensureAllAuthorized(result.hits.hits, operation);
      result.hits.hits.forEach(item => {
        var _this$auditLogger2;
        return (_this$auditLogger2 = this.auditLogger) === null || _this$auditLogger2 === void 0 ? void 0 : _this$auditLogger2.log((0, _lib.alertAuditEvent)({
          action: _lib.operationAlertAuditActionMap[operation],
          id: item._id,
          ...this.getOutcome(operation)
        }));
      });
      return result;
    } catch (error) {
      const errorMessage = `Unable to retrieve alert details for alert with id of "${id}" or with query "${JSON.stringify(query)}" and operation ${operation} \nError: ${error}`;
      this.logger.error(errorMessage);
      throw _boom.default.notFound(errorMessage);
    }
  }

  /**
   * When an update by ids is requested, do a multi-get, ensure authz and audit alerts, then execute bulk update
   */
  async mgetAlertsAuditOperate({
    alerts,
    operation,
    fieldToUpdate,
    validate
  }) {
    try {
      const mgetRes = await this.ensureAllAlertsAuthorized({
        alerts,
        operation
      });
      const updateRequests = [];
      for (const item of mgetRes.docs) {
        if (validate) {
          // @ts-expect-error doesn't handle error branch in MGetResponse
          validate(item === null || item === void 0 ? void 0 : item._source);
        }
        updateRequests.push([{
          update: {
            _index: item._index,
            _id: item._id
          }
        }, {
          doc: {
            // @ts-expect-error doesn't handle error branch in MGetResponse
            ...fieldToUpdate(item === null || item === void 0 ? void 0 : item._source)
          }
        }]);
      }
      const bulkUpdateRequest = updateRequests.flat();
      const bulkUpdateResponse = await this.esClient.bulk({
        refresh: 'wait_for',
        body: bulkUpdateRequest
      });
      return bulkUpdateResponse;
    } catch (exc) {
      this.logger.error(`error in mgetAlertsAuditOperate ${exc}`);
      throw exc;
    }
  }

  /**
   * When an update by ids is requested, do a multi-get, ensure authz and audit alerts, then execute bulk update
   */
  async mgetAlertsAuditOperateStatus({
    alerts,
    status,
    operation
  }) {
    return this.mgetAlertsAuditOperate({
      alerts,
      operation,
      fieldToUpdate: source => this.getAlertStatusFieldUpdate(source, status)
    });
  }
  async buildEsQueryWithAuthz(query, id, alertSpaceId, operation, config, ruleTypeIds, consumers) {
    try {
      const authzFilter = await (0, _lib2.getAuthzFilter)(this.authorization, operation);
      const spacesFilter = (0, _lib2.getSpacesFilter)(alertSpaceId);
      const ruleTypeIdsFilter = (0, _get_rule_type_ids_filter.getRuleTypeIdsFilter)(ruleTypeIds);
      const consumersFilter = (0, _get_consumers_filter.getConsumersFilter)(consumers);
      let esQuery;
      if (id != null) {
        esQuery = {
          query: `_id:${id}`,
          language: 'kuery'
        };
      } else if (typeof query === 'string') {
        esQuery = {
          query,
          language: 'kuery'
        };
      } else if (query != null && typeof query === 'object') {
        esQuery = [];
      }
      const builtQuery = (0, _esQuery.buildEsQuery)(undefined, esQuery == null ? {
        query: ``,
        language: 'kuery'
      } : esQuery, [authzFilter, spacesFilter, ruleTypeIdsFilter, consumersFilter], config);
      if (query != null && typeof query === 'object') {
        return {
          ...builtQuery,
          bool: {
            ...builtQuery.bool,
            must: [...builtQuery.bool.must, query]
          }
        };
      }
      return builtQuery;
    } catch (exc) {
      this.logger.error(exc);
      throw _boom.default.expectationFailed(`buildEsQueryWithAuthz threw an error: unable to get authorization filter \n ${exc}`);
    }
  }

  /**
   * Executes a search after to find alerts with query (+ authz filter)
   */
  async queryAndAuditAllAlerts({
    index,
    query,
    operation
  }) {
    let lastSortIds;
    let hasSortIds = true;
    const alertSpaceId = this.spaceId;
    if (alertSpaceId == null) {
      this.logger.error('Failed to acquire spaceId from authorization client');
      return;
    }
    const config = (0, _ruleDataUtils.getEsQueryConfig)();
    const authorizedQuery = await this.buildEsQueryWithAuthz(query, null, alertSpaceId, operation, config);
    while (hasSortIds) {
      try {
        var _result$hits$hits2;
        const result = await this.searchAlerts({
          id: null,
          query,
          index,
          operation,
          lastSortIds
        });
        if (lastSortIds != null && (result === null || result === void 0 ? void 0 : result.hits.hits.length) === 0) {
          return {
            auditedAlerts: true,
            authorizedQuery
          };
        }
        if (result == null) {
          this.logger.error('RESULT WAS EMPTY');
          return {
            auditedAlerts: false,
            authorizedQuery
          };
        }
        if (result.hits.hits.length === 0) {
          this.logger.error('Search resulted in no hits');
          return {
            auditedAlerts: true,
            authorizedQuery
          };
        }
        lastSortIds = (0, _ruleDataUtils.getSafeSortIds)((_result$hits$hits2 = result.hits.hits[result.hits.hits.length - 1]) === null || _result$hits$hits2 === void 0 ? void 0 : _result$hits$hits2.sort);
        if (lastSortIds != null && lastSortIds.length !== 0) {
          hasSortIds = true;
        } else {
          hasSortIds = false;
          return {
            auditedAlerts: true,
            authorizedQuery
          };
        }
      } catch (error) {
        const errorMessage = `queryAndAuditAllAlerts threw an error: Unable to retrieve alerts with query "${query}" and operation ${operation} \n ${error}`;
        this.logger.error(errorMessage);
        throw _boom.default.notFound(errorMessage);
      }
    }
  }

  /**
   * Ensures that the user has access to the alerts
   * for a given operation
   */
  async ensureAllAlertsAuthorized({
    alerts,
    operation
  }) {
    try {
      const mgetRes = await this.esClient.mget({
        docs: alerts.map(({
          id,
          index
        }) => ({
          _id: id,
          _index: index
        }))
      });
      await this.ensureAllAuthorized(mgetRes.docs, operation);
      const ids = mgetRes.docs.map(({
        _id
      }) => _id);
      for (const id of ids) {
        var _this$auditLogger3;
        (_this$auditLogger3 = this.auditLogger) === null || _this$auditLogger3 === void 0 ? void 0 : _this$auditLogger3.log((0, _lib.alertAuditEvent)({
          action: _lib.operationAlertAuditActionMap[operation],
          id,
          ...this.getOutcome(operation)
        }));
      }
      return mgetRes;
    } catch (exc) {
      this.logger.error(`error in ensureAllAlertsAuthorized ${exc}`);
      throw exc;
    }
  }
  async ensureAllAlertsAuthorizedRead({
    alerts
  }) {
    try {
      await this.ensureAllAlertsAuthorized({
        alerts,
        operation: _server.ReadOperations.Get
      });
    } catch (error) {
      this.logger.error(`error authenticating alerts for read access: ${error}`);
      throw error;
    }
  }
  async get({
    id,
    index
  }) {
    try {
      // first search for the alert by id, then use the alert info to check if user has access to it
      const alert = await this.searchAlerts({
        id,
        index,
        operation: _server.ReadOperations.Get
      });
      if (alert == null || alert.hits.hits.length === 0) {
        const errorMessage = `Unable to retrieve alert details for alert with id of "${id}" and operation ${_server.ReadOperations.Get}`;
        this.logger.error(errorMessage);
        throw _boom.default.notFound(errorMessage);
      }

      // move away from pulling data from _source in the future
      return {
        ...alert.hits.hits[0]._source,
        _index: alert.hits.hits[0]._index
      };
    } catch (error) {
      this.logger.error(`get threw an error: ${error}`);
      throw error;
    }
  }
  async getAlertSummary({
    gte,
    lte,
    ruleTypeIds,
    consumers,
    filter,
    fixedInterval = '1m'
  }) {
    try {
      var _ref, _responseAlertSum$agg, _responseAlertSum$agg2, _buckets, _responseAlertSum$agg3, _responseAlertSum$agg4, _buckets2, _responseAlertSum$agg5, _responseAlertSum$agg6, _responseAlertSum$agg7;
      const indexToUse = await this.getAuthorizedAlertsIndices(ruleTypeIds);
      if ((0, _lodash.isEmpty)(indexToUse)) {
        throw _boom.default.badRequest('no ruleTypeIds were provided for getting alert summary');
      }

      // first search for the alert by id, then use the alert info to check if user has access to it
      const responseAlertSum = await this.searchAlerts({
        index: (indexToUse !== null && indexToUse !== void 0 ? indexToUse : []).join(),
        operation: _server.ReadOperations.Get,
        aggs: {
          active_alerts_bucket: {
            date_histogram: {
              field: _ruleDataUtils.ALERT_TIME_RANGE,
              fixed_interval: fixedInterval,
              hard_bounds: {
                min: gte,
                max: lte
              },
              extended_bounds: {
                min: gte,
                max: lte
              },
              min_doc_count: 0
            }
          },
          recovered_alerts: {
            filter: {
              term: {
                [_ruleDataUtils.ALERT_STATUS]: _ruleDataUtils.ALERT_STATUS_RECOVERED
              }
            },
            aggs: {
              container: {
                date_histogram: {
                  field: _ruleDataUtils.ALERT_END,
                  fixed_interval: fixedInterval,
                  extended_bounds: {
                    min: gte,
                    max: lte
                  },
                  min_doc_count: 0
                }
              }
            }
          },
          count: {
            terms: {
              field: _ruleDataUtils.ALERT_STATUS
            }
          }
        },
        query: {
          bool: {
            filter: [{
              range: {
                [_ruleDataUtils.ALERT_TIME_RANGE]: {
                  gt: gte,
                  lt: lte
                }
              }
            }, ...(filter ? filter : [])]
          }
        },
        size: 0,
        ruleTypeIds,
        consumers
      });
      let activeAlertCount = 0;
      let recoveredAlertCount = 0;
      ((_ref = responseAlertSum === null || responseAlertSum === void 0 ? void 0 : (_responseAlertSum$agg = responseAlertSum.aggregations) === null || _responseAlertSum$agg === void 0 ? void 0 : (_responseAlertSum$agg2 = _responseAlertSum$agg.count) === null || _responseAlertSum$agg2 === void 0 ? void 0 : _responseAlertSum$agg2.buckets) !== null && _ref !== void 0 ? _ref : []).forEach(b => {
        if (b.key === _ruleDataUtils.ALERT_STATUS_ACTIVE) {
          activeAlertCount = b.doc_count;
        } else if (b.key === _ruleDataUtils.ALERT_STATUS_RECOVERED) {
          recoveredAlertCount = b.doc_count;
        }
      });
      return {
        activeAlertCount,
        recoveredAlertCount,
        activeAlerts: (_buckets = responseAlertSum === null || responseAlertSum === void 0 ? void 0 : (_responseAlertSum$agg3 = responseAlertSum.aggregations) === null || _responseAlertSum$agg3 === void 0 ? void 0 : (_responseAlertSum$agg4 = _responseAlertSum$agg3.active_alerts_bucket) === null || _responseAlertSum$agg4 === void 0 ? void 0 : _responseAlertSum$agg4.buckets) !== null && _buckets !== void 0 ? _buckets : [],
        recoveredAlerts: (_buckets2 = responseAlertSum === null || responseAlertSum === void 0 ? void 0 : (_responseAlertSum$agg5 = responseAlertSum.aggregations) === null || _responseAlertSum$agg5 === void 0 ? void 0 : (_responseAlertSum$agg6 = _responseAlertSum$agg5.recovered_alerts) === null || _responseAlertSum$agg6 === void 0 ? void 0 : (_responseAlertSum$agg7 = _responseAlertSum$agg6.container) === null || _responseAlertSum$agg7 === void 0 ? void 0 : _responseAlertSum$agg7.buckets) !== null && _buckets2 !== void 0 ? _buckets2 : []
      };
    } catch (error) {
      this.logger.error(`getAlertSummary threw an error: ${error}`);
      throw error;
    }
  }
  async update({
    id,
    status,
    _version,
    index
  }) {
    try {
      const alert = await this.searchAlerts({
        id,
        index,
        operation: _server.WriteOperations.Update
      });
      if (alert == null || alert.hits.hits.length === 0) {
        const errorMessage = `Unable to retrieve alert details for alert with id of "${id}" and operation ${_server.ReadOperations.Get}`;
        this.logger.error(errorMessage);
        throw _boom.default.notFound(errorMessage);
      }
      const fieldToUpdate = this.getAlertStatusFieldUpdate(alert === null || alert === void 0 ? void 0 : alert.hits.hits[0]._source, status);
      const response = await this.esClient.update({
        ...(0, _securitysolutionEsUtils.decodeVersion)(_version),
        id,
        index,
        doc: {
          ...fieldToUpdate
        },
        refresh: 'wait_for'
      });
      return {
        ...response,
        _version: (0, _securitysolutionEsUtils.encodeHitVersion)(response)
      };
    } catch (error) {
      this.logger.error(`update threw an error: ${error}`);
      throw error;
    }
  }
  async bulkUpdate({
    ids,
    query,
    index,
    status
  }) {
    // rejects at the route level if more than 1000 id's are passed in
    if (ids != null) {
      const alerts = ids.map(id => ({
        id,
        index
      }));
      return this.mgetAlertsAuditOperateStatus({
        alerts,
        status,
        operation: _server.WriteOperations.Update
      });
    } else if (query != null) {
      try {
        // execute search after with query + authorization filter
        // audit results of that query
        const fetchAndAuditResponse = await this.queryAndAuditAllAlerts({
          query,
          index,
          operation: _server.WriteOperations.Update
        });
        if (!(fetchAndAuditResponse !== null && fetchAndAuditResponse !== void 0 && fetchAndAuditResponse.auditedAlerts)) {
          throw _boom.default.forbidden('Failed to audit alerts');
        }

        // executes updateByQuery with query + authorization filter
        // used in the queryAndAuditAllAlerts function
        const result = await this.esClient.updateByQuery({
          index,
          conflicts: 'proceed',
          script: {
            source: `if (ctx._source['${_technical_rule_data_field_names.ALERT_WORKFLOW_STATUS}'] != null) {
                ctx._source['${_technical_rule_data_field_names.ALERT_WORKFLOW_STATUS}'] = '${status}'
              }
              if (ctx._source.signal != null && ctx._source.signal.status != null) {
                ctx._source.signal.status = '${status}'
              }`,
            lang: 'painless'
          },
          query: fetchAndAuditResponse.authorizedQuery,
          ignore_unavailable: true
        });
        return result;
      } catch (err) {
        this.logger.error(`bulkUpdate threw an error: ${err}`);
        throw err;
      }
    } else {
      throw _boom.default.badRequest('no ids or query were provided for updating');
    }
  }

  /**
   * This function updates the case ids of multiple alerts per index.
   * It is supposed to be used only by Cases.
   * Cases implements its own RBAC. By using this function directly
   * Cases RBAC is bypassed.
   * Plugins that want to attach alerts to a case should use the
   * cases client that does all the necessary cases RBAC checks
   * before updating the alert with the case ids.
   */
  async bulkUpdateCases({
    alerts,
    caseIds
  }) {
    if (alerts.length === 0) {
      throw _boom.default.badRequest('You need to define at least one alert to update case ids');
    }

    /**
     * We do this check to avoid any mget calls or authorization checks.
     * The check below does not ensure that an alert may exceed the limit.
     * We need to also throw in case alert.caseIds + caseIds > MAX_CASES_PER_ALERT.
     * The validateTotalCasesPerAlert function ensures that.
     */
    if (caseIds.length > _ruleDataUtils.MAX_CASES_PER_ALERT) {
      throw _boom.default.badRequest(`You cannot attach more than ${_ruleDataUtils.MAX_CASES_PER_ALERT} cases to an alert`);
    }
    return this.mgetAlertsAuditOperate({
      alerts,
      /**
       * A user with read access to an alert and write access to a case should be able to link
       * the case to the alert (update the alert's data to include the case ids).
       * For that reason, the operation is a read operation.
       */
      operation: _server.ReadOperations.Get,
      fieldToUpdate: source => this.getAlertCaseIdsFieldUpdate(source, caseIds),
      validate: source => this.validateTotalCasesPerAlert(source, caseIds)
    });
  }
  async removeCaseIdFromAlerts({
    caseId,
    alerts
  }) {
    /**
     * We intentionally do not perform any authorization
     * on the alerts. Users should be able to remove
     * cases from alerts when deleting a case or an
     * attachment
     */
    try {
      if (alerts.length === 0) {
        return;
      }
      const painlessScript = `if (ctx._source['${_ruleDataUtils.ALERT_CASE_IDS}'] != null) {
        if (ctx._source['${_ruleDataUtils.ALERT_CASE_IDS}'].contains('${caseId}')) {
          int index = ctx._source['${_ruleDataUtils.ALERT_CASE_IDS}'].indexOf('${caseId}');
          ctx._source['${_ruleDataUtils.ALERT_CASE_IDS}'].remove(index);
        }
      }`;
      const bulkUpdateRequest = [];
      for (const alert of alerts) {
        bulkUpdateRequest.push({
          update: {
            _index: alert.index,
            _id: alert.id
          }
        }, {
          script: {
            source: painlessScript,
            lang: 'painless'
          }
        });
      }
      await this.esClient.bulk({
        refresh: 'wait_for',
        body: bulkUpdateRequest
      });
    } catch (error) {
      this.logger.error(`Error removing case ${caseId} from alerts: ${error}`);
      throw error;
    }
  }
  async removeCaseIdsFromAllAlerts({
    caseIds
  }) {
    /**
     * We intentionally do not perform any authorization
     * on the alerts. Users should be able to remove
     * cases from alerts when deleting a case or an
     * attachment
     */
    try {
      if (caseIds.length === 0) {
        return;
      }
      const index = `${this.ruleDataService.getResourcePrefix()}-*`;
      const query = `${_ruleDataUtils.ALERT_CASE_IDS}: (${caseIds.join(' or ')})`;
      const esQuery = (0, _esQuery.buildEsQuery)(undefined, {
        query,
        language: 'kuery'
      }, []);
      const SCRIPT_PARAMS_ID = 'caseIds';
      const painlessScript = `if (ctx._source['${_ruleDataUtils.ALERT_CASE_IDS}'] != null && ctx._source['${_ruleDataUtils.ALERT_CASE_IDS}'].length > 0 && params['${SCRIPT_PARAMS_ID}'] != null && params['${SCRIPT_PARAMS_ID}'].length > 0) {
        List storedCaseIds = ctx._source['${_ruleDataUtils.ALERT_CASE_IDS}'];
        List caseIdsToRemove = params['${SCRIPT_PARAMS_ID}'];

        for (int i=0; i < caseIdsToRemove.length; i++) {
          if (storedCaseIds.contains(caseIdsToRemove[i])) {
            int index = storedCaseIds.indexOf(caseIdsToRemove[i]);
            storedCaseIds.remove(index);
          }
        }
      }`;
      await this.esClient.updateByQuery({
        index,
        conflicts: 'proceed',
        script: {
          source: painlessScript,
          lang: 'painless',
          params: {
            caseIds
          }
        },
        query: esQuery,
        ignore_unavailable: true
      });
    } catch (err) {
      this.logger.error(`Failed removing ${caseIds} from all alerts: ${err}`);
      throw err;
    }
  }
  async find({
    aggs,
    ruleTypeIds,
    consumers,
    index,
    query,
    search_after: searchAfter,
    size,
    sort,
    track_total_hits: trackTotalHits,
    _source,
    runtimeMappings
  }) {
    try {
      let indexToUse = index;
      if (ruleTypeIds && !(0, _lodash.isEmpty)(ruleTypeIds)) {
        const tempIndexToUse = await this.getAuthorizedAlertsIndices(ruleTypeIds);
        if (!(0, _lodash.isEmpty)(tempIndexToUse)) {
          indexToUse = (tempIndexToUse !== null && tempIndexToUse !== void 0 ? tempIndexToUse : []).join();
        }
      }
      const alertsSearchResponse = await this.searchAlerts({
        ruleTypeIds,
        consumers,
        query,
        aggs,
        _source,
        track_total_hits: trackTotalHits,
        size,
        index: indexToUse,
        operation: _server.ReadOperations.Find,
        sort,
        lastSortIds: searchAfter,
        runtimeMappings
      });
      if (alertsSearchResponse == null) {
        const errorMessage = `Unable to retrieve alert details for alert with query and operation ${_server.ReadOperations.Find}`;
        this.logger.error(errorMessage);
        throw _boom.default.notFound(errorMessage);
      }
      return alertsSearchResponse;
    } catch (error) {
      this.logger.error(`find threw an error: ${error}`);
      throw error;
    }
  }

  /**
   * Performs a `find` query to extract aggregations on alert groups
   */
  async getGroupAggregations({
    ruleTypeIds,
    consumers,
    groupByField,
    aggregations,
    filters,
    pageIndex,
    pageSize,
    sort = [{
      unitsCount: {
        order: 'desc'
      }
    }]
  }) {
    var _searchResult$aggrega;
    const uniqueValue = (0, _uuid.v4)();
    if (pageIndex > _constants.MAX_ALERTS_PAGES) {
      throw _boom.default.badRequest(`The provided pageIndex value is too high. The maximum allowed pageIndex value is ${_constants.MAX_ALERTS_PAGES}.`);
    }
    if (Math.max(pageIndex, pageIndex * pageSize) > _constants.MAX_PAGINATED_ALERTS) {
      throw _boom.default.badRequest(`The number of documents is too high. Paginating through more than ${_constants.MAX_PAGINATED_ALERTS} documents is not possible.`);
    }
    const searchResult = await this.find({
      ruleTypeIds,
      consumers,
      aggs: {
        groupByFields: {
          terms: {
            field: 'groupByField',
            size: _constants.MAX_ALERTS_GROUPING_QUERY_SIZE
          },
          aggs: {
            unitsCount: {
              value_count: {
                field: 'groupByField'
              }
            },
            bucket_truncate: {
              bucket_sort: {
                sort,
                from: pageIndex * pageSize,
                size: pageSize
              }
            },
            ...(aggregations !== null && aggregations !== void 0 ? aggregations : {})
          }
        },
        unitsCount: {
          value_count: {
            field: 'groupByField'
          }
        },
        groupsCount: {
          cardinality: {
            field: 'groupByField'
          }
        }
      },
      query: {
        bool: {
          filter: filters
        }
      },
      runtimeMappings: {
        groupByField: {
          type: 'keyword',
          script: {
            source:
            // When size()==0, emits a uniqueValue as the value to represent this group  else join by uniqueValue.
            "if (!doc.containsKey(params['selectedGroup']) || doc[params['selectedGroup']].size()==0) { emit(params['uniqueValue']) }" +
            // Else, join the values with uniqueValue. We cannot simply emit the value like doc[params['selectedGroup']].value,
            // the runtime field will only return the first value in an array.
            // The docs advise that if the field has multiple values, "Scripts can call the emit method multiple times to emit multiple values."
            // However, this gives us a group for each value instead of combining the values like we're aiming for.
            // Instead of .value, we can retrieve all values with .join().
            // Instead of joining with a "," we should join with a unique value to avoid splitting a value that happens to contain a ",".
            // We will format into a proper array in parseGroupingQuery .
            " else { emit(doc[params['selectedGroup']].join(params['uniqueValue']))}",
            params: {
              selectedGroup: groupByField,
              uniqueValue
            }
          }
        }
      },
      size: 0,
      _source: false
    });
    // Replace artificial uuid values with '--' in null-value buckets and mark them with `isNullGroup = true`
    const groupsAggregation = (_searchResult$aggrega = searchResult.aggregations) === null || _searchResult$aggrega === void 0 ? void 0 : _searchResult$aggrega.groupByFields;
    if (groupsAggregation) {
      var _groupsAggregation$bu;
      const buckets = Array.isArray(groupsAggregation === null || groupsAggregation === void 0 ? void 0 : groupsAggregation.buckets) ? groupsAggregation.buckets : Object.values((_groupsAggregation$bu = groupsAggregation === null || groupsAggregation === void 0 ? void 0 : groupsAggregation.buckets) !== null && _groupsAggregation$bu !== void 0 ? _groupsAggregation$bu : {});
      buckets.forEach(bucket => {
        if (bucket.key === uniqueValue) {
          bucket.key = '--';
          bucket.isNullGroup = true;
        }
      });
    }
    return searchResult;
  }
  async getAuthorizedAlertsIndices(ruleTypeIds) {
    try {
      const authorizedRuleTypes = await this.authorization.getAllAuthorizedRuleTypesFindOperation({
        authorizationEntity: _server.AlertingAuthorizationEntity.Alert,
        ruleTypeIds
      });
      const indices = this.getAlertIndicesAlias(Array.from(authorizedRuleTypes.keys()).map(id => id), this.spaceId);
      return indices;
    } catch (exc) {
      const errMessage = `getAuthorizedAlertsIndices failed to get authorized rule types: ${exc}`;
      this.logger.error(errMessage);
      throw _boom.default.failedDependency(errMessage);
    }
  }
  async getBrowserFields({
    ruleTypeIds,
    indices,
    metaFields,
    allowNoIndex,
    includeEmptyFields,
    indexFilter
  }) {
    const indexPatternsFetcherAsInternalUser = new _server2.IndexPatternsFetcher(this.esClient);
    const {
      fields
    } = await indexPatternsFetcherAsInternalUser.getFieldsForWildcard({
      pattern: indices,
      metaFields,
      fieldCapsOptions: {
        allow_no_indices: allowNoIndex
      },
      includeEmptyFields,
      indexFilter
    });
    return {
      browserFields: (0, _browser_fields.fieldDescriptorToBrowserFieldMapper)(fields),
      fields
    };
  }
  getRuleTypeIds(ruleTypeIds) {
    // fetch all rule types if no specific rule type Ids are provided
    if (ruleTypeIds.length === 0) {
      const registeredRuleTypes = this.getRuleList();
      if (!registeredRuleTypes) {
        return [];
      }
      return Array.from(registeredRuleTypes.keys());
    }
    return ruleTypeIds;
  }
  async getAlertFields(ruleTypeIds) {
    const allRuleTypesIds = this.getRuleTypeIds(ruleTypeIds);
    const authorizedRuleTypes = await this.authorization.getAllAuthorizedRuleTypesFindOperation({
      authorizationEntity: _server.AlertingAuthorizationEntity.Alert,
      ruleTypeIds: allRuleTypesIds
    });
    const authorizedRuleTypesIds = Array.from(authorizedRuleTypes.keys());
    const [siemRuleTypeIds, otherRuleTypeIds] = (0, _lodash.partition)(authorizedRuleTypesIds, ruleTypeId => (0, _ruleDataUtils.isSiemRuleType)(ruleTypeId));
    const siemIndices = siemRuleTypeIds ? this.getAlertIndicesAlias(siemRuleTypeIds) : [];
    const otherIndices = otherRuleTypeIds ? this.getAlertIndicesAlias(otherRuleTypeIds) : [];
    const indexPatternsFetcherAsInternalUser = new _server2.IndexPatternsFetcher(this.esClient);
    const indexPatternsFetcherAsScoped = new _server2.IndexPatternsFetcher(this.esClientScoped);
    const [specFields, descriptorFields] = await Promise.all([(0, _get_alert_fields_from_index_fetcher.getAlertFieldsFromIndexFetcher)(indexPatternsFetcherAsScoped, siemIndices), (0, _get_alert_fields_from_index_fetcher.getAlertFieldsFromIndexFetcher)(indexPatternsFetcherAsInternalUser, otherIndices)]);
    const uniqueFields = (0, _unique_fields.mergeUniqueFieldsByName)(descriptorFields, specFields);
    const mappedFields = {
      fields: uniqueFields
    };
    return mappedFields;
  }
}
exports.AlertsClient = AlertsClient;