"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useToolsService = exports.useTools = exports.useToolService = exports.useTool = void 0;
var _onechatBrowser = require("@kbn/onechat-browser");
var _reactQuery = require("@tanstack/react-query");
var _react = require("react");
var _query_keys = require("../../query_keys");
var _i18n = require("../../utils/i18n");
var _use_onechat_service = require("../use_onechat_service");
var _use_toasts = require("../use_toasts");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const useToolsService = () => {
  const {
    toolsService
  } = (0, _use_onechat_service.useOnechatServices)();
  const {
    data,
    isLoading,
    error,
    isError
  } = (0, _reactQuery.useQuery)({
    queryKey: _query_keys.queryKeys.tools.all,
    queryFn: () => toolsService.list()
  });
  return {
    tools: data !== null && data !== void 0 ? data : [],
    isLoading,
    error,
    isError
  };
};
exports.useToolsService = useToolsService;
const useToolService = toolId => {
  const {
    toolsService
  } = (0, _use_onechat_service.useOnechatServices)();
  const {
    data: tool,
    isLoading,
    error,
    isError
  } = (0, _reactQuery.useQuery)({
    enabled: !!toolId,
    queryKey: _query_keys.queryKeys.tools.byId(toolId),
    // toolId! is safe because of the enabled check above
    queryFn: () => toolsService.get({
      toolId: toolId
    })
  });
  return {
    tool: tool,
    isLoading,
    error,
    isError
  };
};
exports.useToolService = useToolService;
const useTool = ({
  toolId,
  onLoadingError
}) => {
  const {
    addErrorToast
  } = (0, _use_toasts.useToasts)();
  const {
    tool,
    isLoading,
    error,
    isError
  } = useToolService(toolId);
  (0, _react.useEffect)(() => {
    if (toolId && isError) {
      const formattedError = (0, _onechatBrowser.formatOnechatErrorMessage)(error);
      addErrorToast({
        title: _i18n.labels.tools.loadToolErrorToast(toolId),
        text: formattedError
      });
      onLoadingError === null || onLoadingError === void 0 ? void 0 : onLoadingError(new Error(formattedError));
    }
  }, [isError, error, toolId, addErrorToast, onLoadingError]);
  return {
    tool,
    isLoading,
    error,
    isError
  };
};
exports.useTool = useTool;
const useTools = ({
  onLoadingError
} = {}) => {
  const {
    addErrorToast
  } = (0, _use_toasts.useToasts)();
  const {
    tools,
    isLoading,
    error,
    isError
  } = useToolsService();
  (0, _react.useEffect)(() => {
    if (isError) {
      const formattedError = (0, _onechatBrowser.formatOnechatErrorMessage)(error);
      addErrorToast({
        title: _i18n.labels.tools.loadToolsErrorToast,
        text: formattedError
      });
      onLoadingError === null || onLoadingError === void 0 ? void 0 : onLoadingError(new Error(formattedError));
    }
  }, [isError, error, addErrorToast, onLoadingError]);
  return {
    tools,
    isLoading,
    error,
    isError
  };
};
exports.useTools = useTools;