"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useDeleteToolsService = exports.useDeleteTools = exports.useDeleteToolService = exports.useDeleteTool = void 0;
var _onechatBrowser = require("@kbn/onechat-browser");
var _reactQuery = require("@tanstack/react-query");
var _react = require("react");
var _query_keys = require("../../query_keys");
var _i18n = require("../../utils/i18n");
var _use_onechat_service = require("../use_onechat_service");
var _use_toasts = require("../use_toasts");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const useDeleteToolService = ({
  onSuccess,
  onError
} = {}) => {
  const queryClient = (0, _reactQuery.useQueryClient)();
  const {
    toolsService
  } = (0, _use_onechat_service.useOnechatServices)();
  const {
    mutate,
    mutateAsync,
    isLoading
  } = (0, _reactQuery.useMutation)({
    mutationFn: ({
      toolId
    }) => toolsService.delete({
      toolId
    }),
    onSettled: () => queryClient.invalidateQueries({
      queryKey: _query_keys.queryKeys.tools.all
    }),
    onSuccess,
    onError
  });
  return {
    deleteToolSync: mutate,
    deleteTool: mutateAsync,
    isLoading
  };
};
exports.useDeleteToolService = useDeleteToolService;
const useDeleteToolsService = ({
  onSuccess,
  onError
} = {}) => {
  const queryClient = (0, _reactQuery.useQueryClient)();
  const {
    toolsService
  } = (0, _use_onechat_service.useOnechatServices)();
  const {
    mutate,
    mutateAsync,
    isLoading
  } = (0, _reactQuery.useMutation)({
    mutationFn: ({
      toolIds
    }) => toolsService.bulkDelete(toolIds),
    onSettled: () => queryClient.invalidateQueries({
      queryKey: _query_keys.queryKeys.tools.all
    }),
    onSuccess,
    onError
  });
  return {
    deleteToolsSync: mutate,
    deleteTools: mutateAsync,
    isLoading
  };
};
exports.useDeleteToolsService = useDeleteToolsService;
const useDeleteTool = ({
  onSuccess,
  onError
} = {}) => {
  const {
    addSuccessToast,
    addErrorToast
  } = (0, _use_toasts.useToasts)();
  const [deleteToolId, setDeleteToolId] = (0, _react.useState)(null);
  const onConfirmCallbackRef = (0, _react.useRef)();
  const onCancelCallbackRef = (0, _react.useRef)();
  const isModalOpen = deleteToolId !== null;
  const deleteTool = (0, _react.useCallback)((toolId, {
    onConfirm,
    onCancel
  } = {}) => {
    setDeleteToolId(toolId);
    onConfirmCallbackRef.current = onConfirm;
    onCancelCallbackRef.current = onCancel;
  }, []);
  const handleSuccess = (data, {
    toolId
  }) => {
    if (!data.success) {
      addErrorToast({
        title: _i18n.labels.tools.deleteToolErrorToast(toolId),
        text: (0, _onechatBrowser.formatOnechatErrorMessage)(new Error('Delete operation failed. API returned: { success: false }'))
      });
      return;
    }
    addSuccessToast({
      title: _i18n.labels.tools.deleteToolSuccessToast(toolId)
    });
    setDeleteToolId(null);
  };
  const handleError = (error, {
    toolId
  }) => {
    addErrorToast({
      title: _i18n.labels.tools.deleteToolErrorToast(toolId),
      text: (0, _onechatBrowser.formatOnechatErrorMessage)(error)
    });
  };
  const {
    deleteTool: deleteToolMutation,
    isLoading
  } = useDeleteToolService({
    onSuccess: handleSuccess,
    onError: handleError
  });
  const confirmDelete = (0, _react.useCallback)(async () => {
    var _onConfirmCallbackRef;
    if (!deleteToolId) {
      return;
    }
    await deleteToolMutation({
      toolId: deleteToolId
    }, {
      onSuccess,
      onError
    });
    (_onConfirmCallbackRef = onConfirmCallbackRef.current) === null || _onConfirmCallbackRef === void 0 ? void 0 : _onConfirmCallbackRef.call(onConfirmCallbackRef);
    onConfirmCallbackRef.current = undefined;
  }, [deleteToolId, deleteToolMutation, onSuccess, onError]);
  const cancelDelete = (0, _react.useCallback)(() => {
    var _onCancelCallbackRef$;
    setDeleteToolId(null);
    (_onCancelCallbackRef$ = onCancelCallbackRef.current) === null || _onCancelCallbackRef$ === void 0 ? void 0 : _onCancelCallbackRef$.call(onCancelCallbackRef);
    onCancelCallbackRef.current = undefined;
  }, []);
  return {
    isOpen: isModalOpen,
    isLoading,
    toolId: deleteToolId,
    deleteTool,
    confirmDelete,
    cancelDelete
  };
};
exports.useDeleteTool = useDeleteTool;
const useDeleteTools = ({
  onSuccess,
  onError
} = {}) => {
  const {
    addSuccessToast,
    addErrorToast
  } = (0, _use_toasts.useToasts)();
  const [deleteToolIds, setDeleteToolIds] = (0, _react.useState)([]);
  const isModalOpen = deleteToolIds.length > 0;
  const deleteTools = (0, _react.useCallback)(toolIds => {
    setDeleteToolIds(toolIds);
  }, []);
  const handleSuccess = ({
    results
  }, {
    toolIds
  }) => {
    if (results.some(result => !result.success)) {
      const failedTools = results.filter(result => !result.success).map(result => result.toolId);
      if (failedTools.length === toolIds.length) {
        addErrorToast({
          title: _i18n.labels.tools.bulkDeleteToolsErrorToast(toolIds.length),
          text: (0, _onechatBrowser.formatOnechatErrorMessage)(new Error('Delete operation failed for all tools.'))
        });
      } else {
        addErrorToast({
          title: _i18n.labels.tools.bulkDeleteToolsErrorToast(failedTools.length),
          text: (0, _onechatBrowser.formatOnechatErrorMessage)(new Error('Delete operation failed for some tools: ' + failedTools.join(', ')))
        });
        setDeleteToolIds([]);
      }
      return;
    }
    addSuccessToast({
      title: _i18n.labels.tools.bulkDeleteToolsSuccessToast(toolIds.length)
    });
    setDeleteToolIds([]);
  };
  const handleError = (error, {
    toolIds
  }) => {
    addErrorToast({
      title: _i18n.labels.tools.bulkDeleteToolsErrorToast(toolIds.length),
      text: (0, _onechatBrowser.formatOnechatErrorMessage)(error)
    });
  };
  const {
    deleteTools: deleteToolsMutation,
    isLoading
  } = useDeleteToolsService({
    onSuccess: handleSuccess,
    onError: handleError
  });
  const confirmDelete = (0, _react.useCallback)(async () => {
    if (!deleteToolIds.length) {
      throw new Error('confirmDelete called outside of modal context');
    }
    return await deleteToolsMutation({
      toolIds: deleteToolIds
    }, {
      onSuccess,
      onError
    });
  }, [deleteToolIds, deleteToolsMutation, onSuccess, onError]);
  const cancelDelete = (0, _react.useCallback)(() => {
    setDeleteToolIds([]);
  }, []);
  return {
    isOpen: isModalOpen,
    isLoading,
    toolIds: deleteToolIds,
    deleteTools,
    confirmDelete,
    cancelDelete
  };
};
exports.useDeleteTools = useDeleteTools;