"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useLocalStorage = useLocalStorage;
var _react = require("react");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const LOCAL_STORAGE_UPDATE_EVENT_TYPE = 'customLocalStorage';
function useLocalStorage(key, defaultValue) {
  const defaultValueRef = (0, _react.useRef)(typeof defaultValue === 'function' ? defaultValue() : defaultValue);
  const [value, setValue] = (0, _react.useState)(() => getFromStorage(key, defaultValueRef.current));
  const valueRef = (0, _react.useRef)(value);
  valueRef.current = value;
  const setter = (0, _react.useMemo)(() => {
    return valueOrCallback => {
      const nextValue = typeof valueOrCallback === 'function' ? valueOrCallback(valueRef.current) : valueOrCallback;
      window.localStorage.setItem(key, JSON.stringify(nextValue));

      /*
       * This is necessary to trigger the event listener in the same
       * window context.
       */
      window.dispatchEvent(new window.CustomEvent(LOCAL_STORAGE_UPDATE_EVENT_TYPE, {
        detail: {
          key
        }
      }));
    };
  }, [key]);
  (0, _react.useEffect)(() => {
    function updateValueFromStorage() {
      setValue(getFromStorage(key, defaultValueRef.current));
    }
    function onStorageEvent(event) {
      if (event.key === key) {
        updateValueFromStorage();
      }
    }
    function onCustomLocalStorageEvent(event) {
      var _event$detail;
      if (event instanceof window.CustomEvent && ((_event$detail = event.detail) === null || _event$detail === void 0 ? void 0 : _event$detail.key) === key) {
        updateValueFromStorage();
      }
    }
    window.addEventListener('storage', onStorageEvent);
    window.addEventListener(LOCAL_STORAGE_UPDATE_EVENT_TYPE, onCustomLocalStorageEvent);
    return () => {
      window.removeEventListener('storage', onStorageEvent);
      window.removeEventListener(LOCAL_STORAGE_UPDATE_EVENT_TYPE, onCustomLocalStorageEvent);
    };
  }, [key]);
  return [value, setter];
}
function getFromStorage(keyName, defaultValue) {
  const storedItem = window.localStorage.getItem(keyName);
  if (storedItem !== null) {
    try {
      return JSON.parse(storedItem);
    } catch (err) {
      window.localStorage.removeItem(keyName);
      // eslint-disable-next-line no-console
      console.log(`Unable to decode: ${keyName}`);
    }
  }
  return defaultValue;
}