"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.SparklineFlotChart = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _lodash = require("lodash");
var _jquery = _interopRequireDefault(require("jquery"));
var _constants = require("../../../common/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Helper class for operations done by Sparkline component on its flot chart
 */

const flotOptions = {
  grid: {
    // No grid
    show: false,
    margin: 4,
    // px

    hoverable: true
  },
  // Set series line color
  colors: ['#3b73ac'],
  // Cribbed from components/chart/get_color.js

  series: {
    // No shadow on series lines
    shadowSize: 0,
    // Cribbed from components/chart/get_options.js

    lines: {
      // Set series line width
      lineWidth: 2 // Cribbed from components/chart/get_options.js
    },
    highlightColor: '#3b73ac',
    points: {
      radius: 2
    }
  }
};
function makeData(series = []) {
  const data = [];

  // The actual series to be rendered
  data.push(series);

  // A fake series, containing only the last point from the actual series, to trick flot
  // into showing the "spark" point of the sparkline.
  data.push({
    data: [(0, _lodash.last)(series)],
    points: {
      show: true,
      radius: 2,
      fill: 1,
      fillColor: false
    }
  });
  return data;
}
class SparklineFlotChart {
  constructor(containerElem, initialSeries, _onBrush, _onHover, overrideOptions) {
    (0, _defineProperty2.default)(this, "render", () => {
      this.flotChart = _jquery.default.plot(this.containerElem, this.data, this.options);
    });
    (0, _defineProperty2.default)(this, "setupBrushHandling", onBrush => {
      // Requires `selection` flot plugin
      this.options.selection = {
        mode: 'x',
        color: '#9c9c9c'
      };
      (0, _jquery.default)(this.containerElem).off('plotselected');
      (0, _jquery.default)(this.containerElem).on('plotselected', (_event, range) => {
        onBrush(range);
        this.flotChart.clearSelection();
      });
    });
    (0, _defineProperty2.default)(this, "setupHoverHandling", onHover => {
      const container = (0, _jquery.default)(this.containerElem);
      const debouncedOnHover = (0, _lodash.debounce)((_event, _range, item) => {
        if (item === null) {
          return onHover();
        }
        onHover({
          xValue: item.datapoint[0],
          yValue: item.datapoint[1],
          xPosition: item.pageX - window.pageXOffset,
          yPosition: item.pageY - window.pageYOffset,
          plotTop: container.offset().top,
          plotLeft: container.offset().left,
          plotHeight: container.height(),
          plotWidth: container.width()
        });
      }, _constants.DEBOUNCE_FAST_MS, {
        leading: true
      });
      container.off('plothover');
      container.on('plothover', debouncedOnHover);
    });
    this.containerElem = containerElem;
    this.data = makeData(initialSeries);
    this.options = {
      ...flotOptions,
      ...overrideOptions
    };
    if ((0, _lodash.isFunction)(_onBrush)) {
      this.setupBrushHandling(_onBrush);
    }
    if ((0, _lodash.isFunction)(_onHover)) {
      this.setupHoverHandling(_onHover);
    }
    this.render();
    window.addEventListener('resize', this.render);
  }
  update(series) {
    this.flotChart.setData(makeData(series));
    this.flotChart.setupGrid();
    this.flotChart.draw();
  }
  /**
   * Necessary to prevent a memory leak. Should be called any time
   * the chart is being removed from the DOM
   */
  shutdown() {
    this.flotChart.shutdown();
    const container = (0, _jquery.default)(this.containerElem);
    container.off('plotselected');
    container.off('plothover');
    window.removeEventListener('resize', this.render);
  }
}
exports.SparklineFlotChart = SparklineFlotChart;