"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.updateList = void 0;
var _utils = require("../utils");
var _ = require(".");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const updateList = async ({
  _version,
  id,
  name,
  description,
  esClient,
  listIndex,
  user,
  meta,
  dateNow,
  version,
  isPatch = false
}) => {
  const updatedAt = dateNow !== null && dateNow !== void 0 ? dateNow : new Date().toISOString();
  const list = await (0, _.getList)({
    esClient,
    id,
    listIndex
  });
  if (list == null) {
    return null;
  } else {
    (0, _utils.checkVersionConflict)(_version, list._version);
    const calculatedVersion = version == null ? list.version + 1 : version;
    const params = {
      description,
      meta,
      name,
      updated_at: updatedAt,
      updated_by: user,
      version: calculatedVersion
    };
    const response = await esClient.updateByQuery({
      conflicts: 'proceed',
      index: listIndex,
      query: {
        ids: {
          values: [id]
        }
      },
      refresh: false,
      script: {
        lang: 'painless',
        params: {
          ...params,
          // when assigning undefined in painless, it will remove property and wil set it to null
          // for patch we don't want to remove unspecified value in payload
          assignEmpty: !isPatch
        },
        source: `
          if (params.assignEmpty == true || params.containsKey('description')) {
            ctx._source.description = params.description;
          }
          if (params.assignEmpty == true || params.containsKey('meta')) {
            ctx._source.meta = params.meta;
          }
          if (params.assignEmpty == true || params.containsKey('name')) {
            ctx._source.name = params.name;
          }
          if (params.assignEmpty == true || params.containsKey('version')) {
            ctx._source.version = params.version;
          }
          ctx._source.updated_at = params.updated_at;
          ctx._source.updated_by = params.updated_by;
          // needed for list that were created before migration to data streams
          if (ctx._source.containsKey('@timestamp') == false) {
            ctx._source['@timestamp'] = ctx._source.created_at;
          }
        `
      }
    });
    let updatedOCCVersion;
    if (response.updated) {
      const checkIfListUpdated = async () => {
        const updatedList = await (0, _.getList)({
          esClient,
          id,
          listIndex
        });
        if ((updatedList === null || updatedList === void 0 ? void 0 : updatedList._version) === list._version) {
          throw Error('Document has not been re-indexed in time');
        }
        updatedOCCVersion = updatedList === null || updatedList === void 0 ? void 0 : updatedList._version;
      };
      await (0, _utils.waitUntilDocumentIndexed)(checkIfListUpdated);
    } else {
      throw Error('No list has been updated');
    }
    return {
      '@timestamp': list['@timestamp'],
      _version: updatedOCCVersion,
      created_at: list.created_at,
      created_by: list.created_by,
      description: description !== null && description !== void 0 ? description : list.description,
      deserializer: list.deserializer,
      id,
      immutable: list.immutable,
      meta: isPatch ? meta !== null && meta !== void 0 ? meta : list.meta : meta,
      name: name !== null && name !== void 0 ? name : list.name,
      serializer: list.serializer,
      tie_breaker_id: list.tie_breaker_id,
      type: list.type,
      updated_at: updatedAt,
      updated_by: user,
      version: calculatedVersion
    };
  }
};
exports.updateList = updateList;