"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.MigrateActionRunner = void 0;
exports.bulkMigrateAgentsBatch = bulkMigrateAgentsBatch;
var _uuid = require("uuid");
var _services = require("../../../common/services");
var _errors = require("../../errors");
var _action_runner = require("./action_runner");
var _bulk_action_types = require("./bulk_action_types");
var _actions = require("./actions");
var _crud = require("./crud");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class MigrateActionRunner extends _action_runner.ActionRunner {
  async processAgents(agents) {
    return await bulkMigrateAgentsBatch(this.esClient, this.soClient, agents, this.actionParams);
  }
  getTaskType() {
    return _bulk_action_types.BulkActionTaskType.MIGRATE_RETRY;
  }
  getActionType() {
    return 'MIGRATE';
  }
}
exports.MigrateActionRunner = MigrateActionRunner;
async function bulkMigrateAgentsBatch(esClient, soClient, agents, options) {
  var _options$actionId, _options$total;
  const errors = {};
  const now = new Date().toISOString();
  const agentPolicies = await (0, _crud.getAgentPolicyForAgents)(soClient, agents);
  const protectedAgentPolicies = agentPolicies.filter(agentPolicy => agentPolicy === null || agentPolicy === void 0 ? void 0 : agentPolicy.is_protected);
  const agentsToAction = [];
  agents.forEach(agent => {
    var _agent$components;
    if (agent.policy_id && protectedAgentPolicies.map(policy => policy.id).includes(agent.policy_id)) {
      errors[agent.id] = new _errors.FleetError(`Agent ${agent.id} cannot be migrated because it is protected.`);
    } else if ((_agent$components = agent.components) !== null && _agent$components !== void 0 && _agent$components.some(c => c.type === 'fleet-server')) {
      errors[agent.id] = new _errors.FleetError(`Agent ${agent.id} cannot be migrated because it is a fleet-server.`);
    } else if (!(0, _services.isAgentMigrationSupported)(agent)) {
      var _agent$local_metadata, _agent$local_metadata2, _agent$local_metadata3;
      // Check if it's specifically a containerized agent
      if (((_agent$local_metadata = agent.local_metadata) === null || _agent$local_metadata === void 0 ? void 0 : (_agent$local_metadata2 = _agent$local_metadata.elastic) === null || _agent$local_metadata2 === void 0 ? void 0 : (_agent$local_metadata3 = _agent$local_metadata2.agent) === null || _agent$local_metadata3 === void 0 ? void 0 : _agent$local_metadata3.upgradeable) === false) {
        errors[agent.id] = new _errors.FleetError(`Agent ${agent.id} cannot be migrated because it is containerized.`);
      } else {
        // Otherwise it's a version issue
        errors[agent.id] = new _errors.FleetError(`Agent ${agent.id} cannot be migrated. Migrate action is supported from version ${_services.MINIMUM_MIGRATE_AGENT_VERSION}.`);
      }
    } else {
      agentsToAction.push(agent);
    }
  });
  const actionId = (_options$actionId = options.actionId) !== null && _options$actionId !== void 0 ? _options$actionId : (0, _uuid.v4)();
  const agentIds = agentsToAction.map(agent => agent.id);
  const total = (_options$total = options.total) !== null && _options$total !== void 0 ? _options$total : agentIds.length;
  const spaceId = options.spaceId;
  const namespaces = spaceId ? [spaceId] : [];
  await (0, _actions.createAgentAction)(esClient, soClient, {
    id: actionId,
    agents: agentIds,
    created_at: now,
    type: 'MIGRATE',
    total,
    data: {
      target_uri: options.uri,
      settings: options.settings
    },
    ...(options.enrollment_token && {
      secrets: {
        enrollment_token: options.enrollment_token
      }
    }),
    namespaces
  });
  await (0, _actions.createErrorActionResults)(esClient, actionId, errors, 'agent does not support migration action');
  return {
    actionId
  };
}